//=============================================================================
//
//  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
//  All Rights Reserved.
//  Confidential and Proprietary - Qualcomm Technologies, Inc.
//
//=============================================================================

#include "Embedding.hpp"
#include "Exception.hpp"
#include "GenieEmbedding.h"
#include "Macro.hpp"
#include "Util/HandleManager.hpp"
#include "qualla/detail/json.hpp"

using namespace genie;

GENIE_API
Genie_Status_t GenieEmbeddingConfig_createFromJson(const char* str,
                                                   GenieEmbeddingConfig_Handle_t* configHandle) {
  try {
    GENIE_ENSURE(str, GENIE_STATUS_ERROR_INVALID_ARGUMENT);
    GENIE_ENSURE(configHandle, GENIE_STATUS_ERROR_INVALID_ARGUMENT);
    auto config = std::make_shared<Embedding::Config>(str);
    GENIE_ENSURE(config, GENIE_STATUS_ERROR_MEM_ALLOC);
    *configHandle = genie::Embedding::Config::add(config);
  } catch (const qualla::json::parse_error& e) {
    std::cerr << e.what() << std::endl;
    return GENIE_STATUS_ERROR_JSON_FORMAT;
  } catch (const Exception& e) {
    std::cerr << e.what() << std::endl;
    return e.status();
  } catch (const std::exception& e) {
    std::cerr << e.what() << std::endl;
    return GENIE_STATUS_ERROR_GENERAL;
  }
  return GENIE_STATUS_SUCCESS;
}

GENIE_API
Genie_Status_t GenieEmbeddingConfig_free(const GenieEmbeddingConfig_Handle_t configHandle) {
  try {
    GENIE_ENSURE(configHandle, GENIE_STATUS_ERROR_INVALID_HANDLE);
    {
      // Check if the embedding actually exists
      auto configObj = genie::Embedding::Config::get(configHandle);
      GENIE_ENSURE(configObj, GENIE_STATUS_ERROR_INVALID_HANDLE);
    }
    genie::Embedding::Config::remove(configHandle);
  } catch (const std::exception& e) {
    return GENIE_STATUS_ERROR_GENERAL;
  }
  return GENIE_STATUS_SUCCESS;
}

GENIE_API
Genie_Status_t GenieEmbedding_create(const GenieEmbeddingConfig_Handle_t configHandle,
                                     GenieEmbedding_Handle_t* embeddingHandle) {
  try {
    GENIE_ENSURE(embeddingHandle, GENIE_STATUS_ERROR_INVALID_ARGUMENT);

    // Get config object
    auto configObj = genie::Embedding::Config::get(configHandle);
    GENIE_ENSURE(configObj, GENIE_STATUS_ERROR_INVALID_HANDLE);

    // Create embedding
    auto embedding = std::make_shared<genie::Embedding>(configObj);
    GENIE_ENSURE(embedding, GENIE_STATUS_ERROR_MEM_ALLOC);

    // Create Handle
    *embeddingHandle = genie::Embedding::add(embedding);
  } catch (const std::exception& e) {
    std::cerr << e.what() << std::endl;
    return GENIE_STATUS_ERROR_GENERAL;
  }

  // Return SUCCESS
  return GENIE_STATUS_SUCCESS;
}

GENIE_API
Genie_Status_t GenieEmbedding_generate(const GenieEmbedding_Handle_t embeddingHandle,
                                       const char* queryStr,
                                       const GenieEmbedding_GenerateCallback_t callback,
                                       const void* userData) {
  int32_t status;

  try {
    GENIE_ENSURE(embeddingHandle, GENIE_STATUS_ERROR_INVALID_HANDLE);
    auto embedding = genie::Embedding::get(embeddingHandle);
    GENIE_ENSURE(embedding, GENIE_STATUS_ERROR_INVALID_HANDLE);
    GENIE_ENSURE(queryStr, GENIE_STATUS_ERROR_INVALID_ARGUMENT);
    GENIE_ENSURE(callback, GENIE_STATUS_ERROR_INVALID_ARGUMENT);

    status = embedding->generate(queryStr, callback, userData);
  } catch (const std::exception& e) {
    std::cerr << e.what() << std::endl;
    return GENIE_STATUS_ERROR_GENERAL;
  }

  return status;
}

GENIE_API
Genie_Status_t GenieEmbedding_free(const GenieEmbedding_Handle_t embeddingHandle) {
  try {
    GENIE_ENSURE(embeddingHandle, GENIE_STATUS_ERROR_INVALID_HANDLE);
    {
      // Check if the embedding actually exists
      auto embedding = genie::Embedding::get(embeddingHandle);
      GENIE_ENSURE(embedding, GENIE_STATUS_ERROR_INVALID_HANDLE);
    }
    genie::Embedding::remove(embeddingHandle);
  } catch (const std::exception& e) {
    return GENIE_STATUS_ERROR_GENERAL;
  }
  return GENIE_STATUS_SUCCESS;
}
