//==============================================================================
//
//  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
//  All Rights Reserved.
//  Confidential and Proprietary - Qualcomm Technologies, Inc.
//
//==============================================================================

#include <qualla/engine.hpp>
#include <qualla/detail/kpi.hpp>
#include <qualla/detail/config.hpp>

#include <functional>
#include <iostream>
#include <sstream>
#include <string>
#include <unordered_map>

#include <fmt/format.h>
#include <fmt/ranges.h>

namespace qualla {

Engine::Engine(Context& ctx, const std::string& type, const qualla::json& conf)
    : _type(type), _ctx(ctx), _env(ctx.env()) {
    _env.logger().debug(
            fmt::format("engine-new: {} ctx {} config {}", type, _ctx.name(), conf.dump())
    );

    using qc = qualla::Config;
    _role    = qc::optional<std::string>(conf, "role", "primary");
}

Engine::~Engine() {}

size_t Engine::process(
        const std::vector<int32_t>& tokens,
        const std::vector<int32_t>& attention_map,
        std::vector<float>&         output,
        bool                        output_all
) {
    _env.logger().error(fmt::format("{}-engine does not support attention_map", _type));
    return 0;
}

size_t Engine::process(const std::vector<int32_t>& tokens) {
    // Derived engines should overwrite this to avoid copying logits
    std::vector<float> logits;
    return process(tokens, logits);
}

size_t Engine::process(
        std::vector<uint8_t>&       embeddings,
        const std::vector<int32_t>& attention_map,
        std::vector<float>&         output,
        bool                        output_all
) {
    _env.logger().error(fmt::format("{}-engine does not support embedding as input", _type));
    return 0;
}

bool Engine::updateKV(size_t n_past) {
    _env.logger().error(fmt::format("{}-engine does not support sync", _type));
    return false;
}

bool Engine::updateKV(size_t n_past, const std::vector<bool>& selected) {
    _env.logger().error(fmt::format("{}-engine does not support sync with selected", _type));
    return false;
}

size_t Engine::restore(const std::string& name, bool chooseHigherVariant) {
    _env.logger().error(fmt::format("{}-engine does not support restore", _type));
    return 0;
}

bool Engine::save(const std::string& name) {
    _env.logger().error(fmt::format("{}-engine does not support save", _type));
    return false;
}

void Engine::reset() {
    _env.logger().error(fmt::format("{}-engine does not support reset", _type));
}

bool Engine::load() {
    _env.logger().error(fmt::format("{}-engine does not support dynamic load", _type));
    return 0;
}

bool Engine::unload() {
    _env.logger().error(fmt::format("{}-engine does not support dynamic unload", _type));
    return false;
}

bool Engine::set(qualla::json data) {
    _env.logger().error(fmt::format("{}-engine does not support set()", _type));
    return false;
}

qualla::json Engine::get() {
    _env.logger().error(fmt::format("{}-engine does not support get()", _type));
    return false;
}

bool Engine::cacheEosEmbedding(std::vector<uint8_t>& eosEmbedding) {
    _env.logger().error(fmt::format("{}-engine does not support cache eos embedding", _type));
    return true;
}

size_t Engine::getEmbeddingBufferSize() {
    _env.logger().error(fmt::format("{}-engine does not support embedding vectors", _type));
    return 0;
}

qualla::InputType Engine::getInputType(){
    return qualla::InputType::TOKENS;
}

// Engine KPIs

std::string Engine::KPIs::dump(std::string_view sep) const {
    return fmt::format(
            "load:[{}]{}process:[{}]{}update-kv:[{}]{}unload:[{}]",
            load.dump(),
            sep,
            process.dump(),
            sep,
            update_kv.dump(),
            sep,
            unload.dump()
    );
}

void Engine::KPIs::reset() {
    load.reset();
    process.reset();
    update_kv.reset();
    unload.reset();
}

// Engine registry type string + creator function
using Registry = std::unordered_map<std::string, Engine::Creator>;
static std::unique_ptr<Registry> registry;

void Engine::__register(const std::string& type, Creator func) {
    if (!registry) registry = std::make_unique<Registry>();

    Registry& r = *registry;
    r[type]     = func;
}

std::unique_ptr<Engine> Engine::create(Context& ctx, const qualla::json& conf) {
    using qc         = qualla::Config;
    
    std::string type = qc::mandatory<std::string>(conf, "type");


    if (!registry) throw std::runtime_error(type + ": engine not found");

    Registry& r = *registry;


    if (!r.contains(type)) throw std::runtime_error(type + ": engine not found");


    return std::unique_ptr<Engine>(r[type](ctx, conf));
}

std::unique_ptr<Engine> Engine::create(Context& ctx, std::istream& json_stream) {
    return create(ctx, json::parse(json_stream));
}

std::unique_ptr<Engine> Engine::create(Context& ctx, const std::string& json_str) {
    return create(ctx, json::parse(json_str));
}

std::vector<std::string> Engine::list() {
    std::vector<std::string> v;
    if (!registry) return v;

    Registry& r = *registry;

    for (auto k : r)
        v.push_back(k.first);
    return v;
}

bool Engine::applyLoraAdapter(std::string lora_adapter_name) {
    _env.logger().error(fmt::format("{}-engine does not support LoraAdapter", _type));
    return false;
}
bool Engine::applyLoraStrength(std::string tensor_name, float tensor_val) {
    _env.logger().error(fmt::format("{}-engine does not support setLoraStrength", _type));
    return false;
}

} // namespace qualla
