//==============================================================================
//
//  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
//  All Rights Reserved.
//  Confidential and Proprietary - Qualcomm Technologies, Inc.
//
//==============================================================================

#include "dlwrap.hpp"
#include "BackendExtensions.hpp"
#include "NetRunBackend.hpp"

BackendExtensions::BackendExtensions(
        BackendExtensionsConfigs             backendExtensionsConfig,
        void*                                backendLibHandle,
        PerfProfile                          perfProfile,
        std::shared_ptr<ICommandLineManager> clManager,
        bool                                 debug_qnn
)
    : m_backendExtensionsLibPath(backendExtensionsConfig.sharedLibraryPath),
      m_backendExtensionsConfigPath(backendExtensionsConfig.configFilePath),
      m_backendInterface(nullptr), m_isNetRunBackendInterface(false),
      m_createBackendInterfaceFn(nullptr), m_destroyBackendInterfaceFn(nullptr),
      m_backendLibHandle(backendLibHandle), m_perfProfile(perfProfile), m_clManager(clManager),
      m_debugQnn(debug_qnn) {
    (void)m_perfProfile;
}

BackendExtensions::~BackendExtensions() {
    if (nullptr != m_backendInterface) {
        if (m_isNetRunBackendInterface) {
            QNN_DEBUG("Deleting NetRun Backend Interface");
            delete m_backendInterface;
        } else {
            if (nullptr != m_destroyBackendInterfaceFn) {
                QNN_DEBUG("Destroying Backend Interface");
                m_destroyBackendInterfaceFn(m_backendInterface);
            }
        }
    }
}

bool BackendExtensions::loadFunctionPointers() {

    void* libHandle = dlopen(m_backendExtensionsLibPath.c_str(), RTLD_NOW | RTLD_LOCAL);
    if (nullptr == libHandle) {
        QNN_ERROR(
                "Unable to load backend extensions lib: [%s]. dlerror(): [%s]",
                m_backendExtensionsLibPath.c_str(),
                dlerror()
        );
        return false;
    }
    m_createBackendInterfaceFn =
            (CreateBackendInterfaceFnType_t)dlsym(libHandle, "createBackendInterface");
    m_destroyBackendInterfaceFn =
            (DestroyBackendInterfaceFnType_t)dlsym(libHandle, "destroyBackendInterface");
    if (nullptr == m_createBackendInterfaceFn || nullptr == m_destroyBackendInterfaceFn) {
        QNN_ERROR("Unable to find symbols. dlerror(): [%s]", dlerror());
        return false;
    }

    return true;
}

void BackendExtensions::qnnLogCallback(
        const char*    fmt,
        QnnLog_Level_t level,
        uint64_t       timestamp,
        va_list        args
) {
    char        buffer[1024] = "";
    const char* levelStr     = "";
    switch (level) {
    case QNN_LOG_LEVEL_ERROR:
        levelStr = " ERROR ";
        break;
    case QNN_LOG_LEVEL_WARN:
        levelStr = "WARNING";
        break;
    case QNN_LOG_LEVEL_INFO:
        levelStr = "  INFO ";
        break;
    case QNN_LOG_LEVEL_DEBUG:
        levelStr = " DEBUG ";
        break;
    case QNN_LOG_LEVEL_VERBOSE:
        levelStr = "VERBOSE";
        break;
    case QNN_LOG_LEVEL_MAX:
        levelStr = "UNKNOWN";
        break;
    }

    int pos = snprintf(
            buffer, sizeof(buffer), "QNN: [%s] time=%lu:", levelStr, (unsigned long)timestamp
    );
    vsnprintf(buffer + pos, sizeof(buffer) - pos, fmt, args);
    printf("%s", buffer);
}

bool BackendExtensions::initialize() {

    QNN_DEBUG("DEBUG: m_backendExtensionsLibPath=%s\n", m_backendExtensionsLibPath.c_str());
    QNN_DEBUG("DEBUG: m_backendExtensionsConfigPath=%s\n", m_backendExtensionsConfigPath.c_str());
    if (m_backendExtensionsLibPath.empty() && m_backendExtensionsConfigPath.empty()) {
        QNN_WARN("No BackendExtensions lib provided; initializing NetRunBackend Interface");
        m_isNetRunBackendInterface = true;
        m_backendInterface         = new NetRunBackend();
    } else {
        QNN_DEBUG("Loading supplied backend extensions lib.");
        QNN_DEBUG("Backend extensions lib path: %s", m_backendExtensionsLibPath.c_str());
        if (m_backendExtensionsConfigPath.empty()) {
            QNN_DEBUG("Backend extensions lib specified without a config file.");
        } else {
            QNN_DEBUG("Backend extensions config path: %s", m_backendExtensionsConfigPath.c_str());
        }
        if (!loadFunctionPointers()) {
            QNN_ERROR("Failed to load function pointers.");
            return false;
        }
        if (nullptr != m_createBackendInterfaceFn) {
            m_backendInterface = m_createBackendInterfaceFn();
        }
    }
    if (nullptr == m_backendInterface) {
        QNN_ERROR("Unable to load backend extensions interface.");
        return false;
    }
    if (m_debugQnn) {
        if (!(m_backendInterface->setupLogging(BackendExtensions::qnnLogCallback, QNN_LOG_LEVEL_VERBOSE))) {
            QNN_WARN("Unable to initialize logging in backend extensions.");
        }
    }
    if (!m_backendInterface->initialize(m_backendLibHandle)) {
        QNN_ERROR("Unable to initialize backend extensions interface.");
        return false;
    }
    if (!m_backendInterface->setPerfProfile(m_perfProfile)) {
        QNN_WARN("Unable to set perf profile in  backend extensions interface.");
        //return false;
    }
    if (!m_backendInterface->loadConfig(m_backendExtensionsConfigPath)) {
        QNN_ERROR("Unable to load backend extensions interface config.");
        return false;
    }

    if ((m_clManager != nullptr) && !m_backendInterface->loadCommandLineArgs(m_clManager)) {
        QNN_ERROR("Unable to load backend extensions' command line arguments.");
        return false;
    }

    return true;
}

IBackend* BackendExtensions::interface() {
    return m_backendInterface;
}
