//  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
//  Confidential & Proprietary - Qualcomm Technologies, Inc. ("QTI")

#include <cassert>
#include <cstring>
#include <fstream>
#include <set>
#include <sstream>

#include "fmt/format.h"
#include "fmt/ranges.h"
#include "fp16/fp16.h"
#include "gpu-model.hpp"
#include "qualla/detail/cache-file.hpp"
#include "qualla/detail/timer.hpp"
#include "qualla/env.hpp"

namespace fs = std::filesystem;

static constexpr uint32_t g_magicNum = 0xC0DE;

#define __INFO(__fmt, ...)  _env.logger().post(Logger::INFO, fmt::format(__fmt, ##__VA_ARGS__));
#define __WARN(__fmt, ...)  _env.logger().post(Logger::WARN, fmt::format(__fmt, ##__VA_ARGS__));
#define __ERROR(__fmt, ...) _env.logger().post(Logger::ERROR, fmt::format(__fmt, ##__VA_ARGS__));
#define __KPIS(__fmt, ...) \
  _env.logger().post(Logger::ENGINE_KPIS, [&]() { return fmt::format(__fmt, ##__VA_ARGS__); });
#define __DEBUG(__fmt, ...) \
  _env.logger().post(Logger::ENGINE_DEBUG, [&]() { return fmt::format(__fmt, ##__VA_ARGS__); });
#define __TRACE(__fmt, ...) \
  _env.logger().post(Logger::ENGINE_TRACE, [&]() { return fmt::format(__fmt, ##__VA_ARGS__); });

namespace qualla {

QnnGpuModel::QnnGpuModel(Env& env, const Params& params)
    : _env(env), _modelBaseDir(params.model_basedir) {
  // Initialize _qnnApi
  _qnnApi = std::unique_ptr<QnnApi>(new QnnApi());

  _ctxSize  = params.ctx_size;
  _numHeads = params.num_heads;
  _headDim  = params.head_dim;
#ifdef _WIN32
  _useDmabufIo = false;
#else
  _useDmabufIo = true;
#endif
  // Set up filename list for context binaries.
  for (auto& i : params.model_list) {
    fs::path model_path = _modelBaseDir / fs::path(i);
    if (!fs::is_regular_file(model_path)) {
      __ERROR("Qnn-Gpu-Model : Can't access model file : {}", model_path.string());
      throw std::runtime_error("Qnn-Gpu-Model : Can't access model file : " + model_path.string());
    }
    _modelList.push_back(model_path.string());
  }
}

QnnGpuModel::~QnnGpuModel() { __INFO("Qnn-Gpu-Model : model destruct complete"); }

// Given a filename, initializeModel load and initializes QNN runtime libraries and the model
bool QnnGpuModel::initializeModel(void) {
  qualla::Timer start;

  __INFO("Qnn-Gpu-Model : Model Init Start");

  const std::string backend = "libQnnGpu.so";

  __INFO("Backend Library : {}", backend);
  __INFO("Model Files : {}", _modelList);

  if (!_qnnApi->initialize(backend, _modelList)) {
    __ERROR("Qnn-Api : Initialization Failed!");
    return false;
  }

  // Initialize QNN IO Tensor
  if (_useDmabufIo) {
    _ioTensor =
        std::unique_ptr<IOTensor>(new IOTensor(BufferAlloc::DMABUF, _qnnApi->getQnnInterfaceVer()));
  } else {
    _ioTensor = std::unique_ptr<IOTensor>(
        new IOTensor(BufferAlloc::DEFAULT, _qnnApi->getQnnInterfaceVer()));
  }
  _numGraphs = _qnnApi->getGraphsCount();
  __INFO("Qnn-Gpu-Model : initialized with {} graph(s)", _numGraphs);

  GraphInfo_t** graphs_info = _qnnApi->getGraphsInfo();
  for (size_t graphIdx = 0; graphIdx < _numGraphs; graphIdx++) {
    GraphInfo_t* const graphInfo = graphs_info[graphIdx];
    char* graphName              = graphInfo->graphName;
    std::string graphStr         = std::string(graphName);

    _modelOrder.push_back(graphStr);
  }
  __INFO("Qnn-Gpu-Model : model init complete: {} usec", start.elapsed_usec());

  return true;
}

// Once the model has been loaded, initialize IO Tensors
// _ioTensors is initialized by the context for now
bool QnnGpuModel::initializeIOTensors() {
  qualla::Timer start;

  // For QNN-GPU, we have only one context per model.
  bool status = _ioTensor->initialize(_qnnApi->getContexts().back());
  if (!status) {
    __ERROR("Qnn-Gpu-Model : failure to initialize IOTensor");
    return false;
  }
  // Getting graph info, Hardcoding single graph for now.
  GraphInfo_t** const& graphsInfo = _qnnApi->getGraphsInfo();

  for (size_t graphIdx = 0; graphIdx < _numGraphs; graphIdx++) {
    GraphInfo_t* const& graphInfo = graphsInfo[graphIdx];
    std::string graphName         = std::string(graphInfo->graphName);

    __DEBUG("Qnn-Gpu-Model : numInputTensors {} numOutputTensors {}",
            graphInfo->numInputTensors,
            graphInfo->numOutputTensors);
    // Setup Inputs
    {
      std::unordered_map<std::string, size_t> inputTensorsSize;
      for (size_t tensorIdx = 0; tensorIdx < graphInfo->numInputTensors; tensorIdx++) {
        std::string tensorName;
        std::vector<size_t> tensorDims;
        auto& tensor = graphInfo->inputTensors[tensorIdx];
        _qnnApi->getTensorNameAndShape(tensorName, tensorDims, tensor);
        auto dims                    = QnnUtils::Dims(tensorDims);
        inputTensorsSize[tensorName] = dims.getSize();
        __DEBUG("Qnn-Gpu-Model : Input Tensor Info {} {} {} {}",
                tensorIdx,
                tensorName,
                tensorDims,
                inputTensorsSize[tensorName]);
        std::vector<QnnUtils::QuantParam> quantParams;
        if (!_qnnApi->getTensorQuantParams(&tensor, quantParams)) {
          quantParams.emplace_back(0, 0);
        }

        std::shared_ptr<QnnUtils::Tensor> tensorUtil =
            std::shared_ptr<QnnUtils::Tensor>(new (std::nothrow) QnnUtils::Tensor);
        tensorUtil->dims                   = dims;
        tensorUtil->dtype                  = QNN_TENSOR_GET_DATA_TYPE(tensor);
        tensorUtil->quantParam             = quantParams;
        _inputSpecs[graphName][tensorName] = tensorUtil;
      }

      Qnn_Tensor_t* tensor_bank = nullptr;
      std::unordered_map<std::string, void*> tensor_ptr_map;
      if (true != _ioTensor->setupInputTensors(&tensor_bank,
                                               tensor_ptr_map,
                                               *graphInfo,
                                               inputTensorsSize,
                                               _qnnApi->getContexts()[graphIdx],
                                               false)) {
        QNN_ERROR("Qnn-Gpu-Model : Error in setting up Input Tensors for graph %s",
                  graphName.c_str());
        return false;
      }

      _inputTensors[graphName] = tensor_bank;
      for (auto& [tensorName, tensor_ptr] : tensor_ptr_map) {
        _inputSpecs[graphName][tensorName]->tensor = (Qnn_Tensor_t*)tensor_ptr;
      }
      __DEBUG("Qnn-Gpu-Model : Input Tensor Allocated for {}", graphName);
    }

    // Setup Outputs
    {
      std::unordered_map<std::string, size_t> outputTensorsSize;
      std::unordered_map<std::string, Qnn_Tensor_t*> sharedTensorMap;
      for (size_t tensorIdx = 0; tensorIdx < graphInfo->numOutputTensors; tensorIdx++) {
        std::string tensorName;
        std::vector<size_t> tensorDims;

        auto& tensor = graphInfo->outputTensors[tensorIdx];
        _qnnApi->getTensorNameAndShape(tensorName, tensorDims, tensor);

        if (tensorName.starts_with("past_")) {
          std::string tensorInName    = tensorName.substr(0, tensorName.size() - 3) + "in";
          sharedTensorMap[tensorName] = _inputSpecs[graphName][tensorInName]->tensor;

          // Update Gpu _kvCache
          auto [type, layer_id] = parseKVTensorName(tensorName);
          _kvCache.push_back(
              GpuKVCache((type == 1), layer_id, _inputSpecs[graphName][tensorInName].get()));
        }
        std::vector<QnnUtils::QuantParam> quantParams;
        if (!_qnnApi->getTensorQuantParams(&tensor, quantParams)) {
          quantParams.emplace_back(0, 0);
        }

        auto dims                     = QnnUtils::Dims(tensorDims);
        outputTensorsSize[tensorName] = dims.getAlignedSize();

        __DEBUG("Qnn-Gpu-Model : Output Tensor Info {} {} {} {}",
                tensorIdx,
                tensorName,
                tensorDims,
                outputTensorsSize[tensorName]);
        std::shared_ptr<QnnUtils::Tensor> tensorUtil =
            std::shared_ptr<QnnUtils::Tensor>(new (std::nothrow) QnnUtils::Tensor);
        tensorUtil->dims                    = dims;
        tensorUtil->dtype                   = QNN_TENSOR_GET_DATA_TYPE(tensor);
        tensorUtil->quantParam              = quantParams;
        _outputSpecs[graphName][tensorName] = tensorUtil;
      }

      Qnn_Tensor_t* tensor_bank = nullptr;
      std::unordered_map<std::string, void*> tensor_ptr_map;
      if (_ioTensor->getBufferAllocType() == BufferAlloc::DMABUF) {
        if (true != _ioTensor->setupOutputWithSharedTensors(&tensor_bank,
                                                            tensor_ptr_map,
                                                            *graphInfo,
                                                            outputTensorsSize,
                                                            _qnnApi->getContexts()[graphIdx],
                                                            sharedTensorMap)) {
          QNN_ERROR("Qnn-Gpu-Model : Error in setting up Output Tensors for graph %s",
                    graphName.c_str());
          return false;
        }
      } else {
        if (true != _ioTensor->setupOutputTensors(&tensor_bank,
                                                  tensor_ptr_map,
                                                  *graphInfo,
                                                  outputTensorsSize,
                                                  _qnnApi->getContexts()[graphIdx],
                                                  false)) {
          QNN_ERROR("Qnn-Gpu-Model : Error in setting up Input Tensors for graph %s",
                    graphName.c_str());
          return false;
        }
      }

      _outputTensors[graphName] = tensor_bank;
      for (auto& [tensorName, tensor_ptr] : tensor_ptr_map) {
        _outputSpecs[graphName][tensorName]->tensor = (Qnn_Tensor_t*)tensor_ptr;
      }

      __DEBUG("Qnn-Gpu-Model : Output Tensor Allocated {} {}", graphName, _outputTensors.size());
    }
  }
  auto stop = std::chrono::steady_clock::now();
  return true;
}

bool QnnGpuModel::initializeTensorPointers() {
  auto inputSpec  = _inputSpecs[_modelOrder.back()];
  auto outputSpec = _outputSpecs[_modelOrder.back()];

  t_inputIds    = inputSpec[INPUT_IDS].get();
  t_attnMask    = inputSpec[ATTN_MASK].get();
  t_positionIds = inputSpec[POS_IDS].get();
  t_logits      = outputSpec[LOGITS].get();

  auto status = !(t_inputIds == nullptr || t_attnMask == nullptr || t_positionIds == nullptr ||
                  t_logits == nullptr);

  if (!status) {
    __ERROR("Qnn-Gpu-Model : error in setting up named tensor pointers for llama.");
    return false;
  }
  return true;
}

bool QnnGpuModel::validateModel() {
  // Validating context Size.
  size_t numInputs    = t_inputIds->dims.getNumElements();
  size_t dimMask      = t_attnMask->dims.getNumElements();
  size_t modelCtxSize = dimMask / numInputs;

  if (modelCtxSize != _ctxSize) {
    __ERROR("Qnn-Gpu-Model : Invalid Context Size {} {}.", modelCtxSize, _ctxSize);
    return false;
  }
  return true;
}

void QnnGpuModel::setupInputTensors(const std::vector<int32_t>& tokens) {
  auto start = std::chrono::steady_clock::now();

  if (tokens.size() > _ctxSize) {
    std::string errMsg = "Called inference with more tokens than model supports: ";
    errMsg += std::to_string(tokens.size()) + " vs. " + std::to_string(_ctxSize);
    throw std::runtime_error(errMsg);
  }

  // Setup 1. input_ids
  // Index of input tokens in the embedding vocabulary
  uint32_t* inputIdBuffer = (uint32_t*)getBuffer(t_inputIds);
  if (inputIdBuffer) {
    if (_useDmabufIo) {
      _ioTensor->beforeWriteToBuffer(t_inputIds->tensor);
    }
    inputIdBuffer[0] = tokens[0];
    if (_useDmabufIo) {
      _ioTensor->afterWriteToBuffer(t_inputIds->tensor);
    }
  }

  // Setup 2. attention_mask
  // Mask to avoid performing attention of padding.
  uint32_t* attnMaskBuffer = (uint32_t*)getBuffer(t_attnMask);
  if (attnMaskBuffer) {
    if (_useDmabufIo) {
      _ioTensor->beforeWriteToBuffer(t_attnMask->tensor);
    }
    attnMaskBuffer[_numTokensProcessed] = 1;
    if (_useDmabufIo) {
      _ioTensor->afterWriteToBuffer(t_attnMask->tensor);
    }
  }

  // Setup 3. position_ids
  // Indices of positions of each input tokens in position embeddings.
  uint32_t* positionIdBuffer = (uint32_t*)getBuffer(t_positionIds);
  if (positionIdBuffer) {
    if (_useDmabufIo) {
      _ioTensor->beforeWriteToBuffer(t_positionIds->tensor);
    }
    positionIdBuffer[0] = (uint32_t)(_numTokensProcessed);
    if (_useDmabufIo) {
      _ioTensor->afterWriteToBuffer(t_positionIds->tensor);
    }
  }

  auto stop = std::chrono::steady_clock::now();
}

template <class T1, class T2>
inline bool QnnGpuModel::executeModel(T1& input, T2& output, std::string graphName) {
  bool ret = _qnnApi->graphExecute(input, output, graphName, timeLogs);
  if (ret != true) {
    QNN_ERROR("Qnn-Gpu-Model : Error executing inference: %d for graph %s", ret, graphName.c_str());
    return false;
  }
  QNN_DEBUG("Qnn-Gpu-Model : Execute finished for graph %s", graphName.c_str());
  return true;
}

bool QnnGpuModel::runInferenceHelper(std::vector<std::string>& exec_models,
                                     int32_t* wait_time_total,
                                     int32_t* exec_time_total,
                                     bool pipeline_kv_update,
                                     size_t update_size) {
  int32_t exec_time = 0;
  int32_t wait_time = 0;
  for (auto& graphName : exec_models) {
    {
      auto start_time = std::chrono::steady_clock::now();
      Qnn_Tensor_t* inputTensors;
      Qnn_Tensor_t* outputTensors;
      try {
        inputTensors  = _inputTensors[graphName];
        outputTensors = _outputTensors[graphName];
      } catch (std::exception e) {
        __DEBUG("Qnn-Gpu-Model : Could not find tensors %s", graphName.c_str());
        return false;
      }
      bool status = executeModel(inputTensors, outputTensors, graphName);
      if (!status) {
        return false;
      }
      auto end_time = std::chrono::steady_clock::now();
      exec_time += static_cast<int32_t>(
          std::chrono::duration_cast<std::chrono::microseconds>(end_time - start_time).count());
    }
  }

  *exec_time_total += exec_time;
  *wait_time_total += wait_time;
  return true;
}

size_t QnnGpuModel::runInference(const std::vector<int32_t>& tokens,
                                 std::vector<float>& logits,
                                 bool logits_all) {
  auto start            = std::chrono::steady_clock::now();
  int32_t totalWaitTime = 0;
  int32_t totalExecTime = 0;

  // Setup inputs for inference
  auto& execModels = _modelOrder;
  int numIters     = tokens.size();
  for (int i = 0; i < numIters; i++) {
    if (numIters > 1) {
      __DEBUG("Qnn-Gpu-Model : Prompt Processing {} of {} tokens", i + 1, numIters);
    } else {
      __DEBUG("Qnn-Gpu-Model : Token Generation {} of {} tokens", i + 1, numIters);
    }
    std::vector<int32_t> curr_tokens;
    curr_tokens.push_back(tokens[i]);
    setupInputTensors(curr_tokens);
    bool status =
        runInferenceHelper(execModels, &totalWaitTime, &totalExecTime, false, tokens.size());
    if (!status) {
      return 0;
    }
    processLogits(logits, logits_all);

    // Update the numProcessTokens to updated with Accepted Tokens.
    _numTokensProcessed++;
  }

  auto stop = std::chrono::steady_clock::now();
  timeLogs["Run Inference (cpp) "].first += static_cast<double>(
      std::chrono::duration_cast<std::chrono::microseconds>(stop - start).count());
  timeLogs["Run Inference (cpp) "].second++;
  QNN_DEBUG("[TIME] Wait[%d] Exec[%d]\n", totalWaitTime, totalExecTime);
  if (!logits_all) {
    return 1;
  }
  return tokens.size();
}

// Parse KV$ Tensor names here - supports past_{key,value}_{layer_idx}[_h0]_{in,out}
std::tuple<int, int> QnnGpuModel::parseKVTensorName(std::string name) {
  if (!name.starts_with("past_")) return {0, 0};

  const bool is_key = name.starts_with("past_key");
  const size_t pos0 = (is_key) ? 9 : 11;  // "past_key_" OR "past_value_"
  const size_t pos1 = name.find('_', pos0);

  int layer_idx = static_cast<int>(std::stoi(name.substr(pos0, pos1 - pos0)));

  return std::make_tuple(is_key ? 1 : 2, layer_idx);
}

size_t QnnGpuModel::loadKVCache(const std::string& load_path) {
  std::ifstream fs(load_path, std::ios::in | std::ios::binary);
  if (fs.fail()) {
    __ERROR("Qnn-Gpu-Model : loadKVCache errror reading file {}", load_path);
    return 0;
  }

  CacheFileSpec spec;
  fs.read((char*)&spec, sizeof(spec));
  if (spec.magic != g_magicNum) {
    __ERROR("Qnn-Gpu-Model : loadKVCache expected {} found {:#x}", g_magicNum, spec.magic);
    return 0;
  }

  // clang-format off
  __INFO("Qnn-Gpu-Model : loadKVCache {{ num_tensors {}, magic {}, dtype {}, n_heads {}, embed_dim {} update_size {} }}",
  spec.num_tensors, spec.magic, int(spec.dtype), spec.n_heads, spec.embed_dim, spec.update_size); fflush(stdout);
  // clang-format on

  _numTokensProcessed = static_cast<size_t>(spec.update_size);
  if (_numTokensProcessed > 0) {
    // Loop over _kvCache tensor and read from file
    for (auto cache : _kvCache) {
      if (_useDmabufIo) {
        _ioTensor->beforeWriteToBuffer(t_inputIds->tensor);
      }
      char* buffer = (char*)getBuffer(cache.tensorUtil);
      if (cache.isKey) {
        // Kye Cache Dims [1, num_heads, head_dim, ctx_size]
        // float16 bits equivalent to uint16_t
        const size_t copySize = _numTokensProcessed;
        const size_t skipSize = _ctxSize;
        for (int i = 0; i < _numHeads; i++) {
          for (int j = 0; j < _headDim; j++) {
            fs.read(buffer, copySize * sizeof(uint16_t));
            buffer += skipSize * sizeof(uint16_t);
          }
        }
      } else {
        // Kye Cache Dims [1, num_heads, ctx_size, head_dim]
        // float16 bits equivalent to uint16_t
        const size_t copySize = _numTokensProcessed * _headDim;
        const size_t skipSize = _ctxSize * _headDim;
        for (int i = 0; i < _numHeads; i++) {
          fs.read(buffer, copySize * sizeof(uint16_t));
          buffer += skipSize * sizeof(uint16_t);
        }
      }
      if (_useDmabufIo) {
        _ioTensor->afterWriteToBuffer(t_inputIds->tensor);
      }
    }
  }
  return _numTokensProcessed;
}

bool QnnGpuModel::saveKVCache(const std::string& save_path) {
  std::ofstream fs(save_path, std::ios::out | std::ios::binary);
  if (fs.fail()) {
    __ERROR("Qnn-Gpu-Model : saveKVCache error opening file : {}", save_path);
    throw std::runtime_error("Failed to write to cache file. Please re-check path");
  }

  const CacheFileSpec::DataType dtype = CacheFileSpec::DataType::FLOAT16_T;

  uint32_t numKVTensors = _kvCache.size();

  // Save the cache file metadata
  CacheFileSpec file_spec(
      numKVTensors, g_magicNum, dtype, 0x0, _numHeads, _headDim, _numTokensProcessed);
  fs.write((char*)&file_spec, sizeof(file_spec));

  // clang-format off
  __INFO("Qnn-Gpu-Model : saveKVCache {{ num_tensors {}, magic {}, dtype {}, n_heads {}, embed_dim {} update_size {} }}",
    numKVTensors, g_magicNum, int(dtype), _numHeads, _headDim, _numTokensProcessed); fflush(stdout);
  // clang-format on

  if (_numTokensProcessed > 0) {
    // Loop over _kvCache tensor and write to file
    for (auto cache : _kvCache) {
      if (_useDmabufIo) {
        _ioTensor->beforeReadFromBuffer(t_inputIds->tensor);
      }
      char* buffer = (char*)getBuffer(cache.tensorUtil);
      if (cache.isKey) {
        // Kye Cache Dims [1, num_heads, head_dim, ctx_size]
        // float16 bits equivalent to uint16_t
        const size_t copySize = _numTokensProcessed;
        const size_t skipSize = _ctxSize;
        for (int i = 0; i < _numHeads; i++) {
          for (int j = 0; j < _headDim; j++) {
            fs.write((char*)buffer, copySize * sizeof(uint16_t));
            buffer += skipSize * sizeof(uint16_t);
          }
        }
      } else {
        // Kye Cache Dims [1, num_heads, ctx_size, head_dim]
        // float16 bits equivalent to uint16_t
        const size_t copySize = _numTokensProcessed * _headDim;
        const size_t skipSize = _ctxSize * _headDim;
        for (int i = 0; i < _numHeads; i++) {
          fs.write((char*)buffer, copySize * sizeof(uint16_t));
          buffer += skipSize;
        }
      }
      if (_useDmabufIo) {
        _ioTensor->afterReadFromBuffer(t_inputIds->tensor);
      }
    }
  }
  fs.flush();
  fs.close();

  return true;
}

size_t QnnGpuModel::processLogits(std::vector<float>& logits, bool logits_all) {
  auto logitsSpec   = _outputSpecs[_modelOrder.back()][LOGITS].get();
  size_t logitsSize = getNumElements(logitsSpec);
  if (_useDmabufIo) {
    _ioTensor->beforeReadFromBuffer(t_inputIds->tensor);
  }
  uint16_t* logitBuf = (uint16_t*)getBuffer(logitsSpec);

  if (!logits_all) {
    logits.clear();
  }
  size_t allocateSize = logits.size() + logitsSize;
  logits.reserve(allocateSize);
  for (auto i = 0; i < logitsSize; ++i) {
    logits.push_back(fp16_ieee_to_fp32_value(logitBuf[i]));
  }
  if (_useDmabufIo) {
    _ioTensor->afterReadFromBuffer(t_inputIds->tensor);
  }

  return logits.size() / logitsSize;
}

bool QnnGpuModel::reset() {
  // Reset Token Counter
  _numTokensProcessed = 0;

  // Reset Attention Mask
  uint32_t* attnMaskBuffer = (uint32_t*)getBuffer(t_attnMask);
  uint32_t attnMaskSize    = getBufferSize(t_attnMask);
  if (attnMaskBuffer) {
    if (_useDmabufIo) {
      _ioTensor->beforeWriteToBuffer(t_attnMask->tensor);
    }
    memset(attnMaskBuffer, 0, attnMaskSize);
    if (_useDmabufIo) {
      _ioTensor->afterWriteToBuffer(t_attnMask->tensor);
    }
  }

  // Reset KV Cache.
  // TODO : Check if mask_neg -100 is enough to remove
  // effect of KV Cache. Test with mask_neg = -float_inf
  for (auto cache : _kvCache) {
    if (_useDmabufIo) {
      _ioTensor->beforeWriteToBuffer(t_inputIds->tensor);
    }
    char* buffer        = (char*)getBuffer(cache.tensorUtil);
    uint32_t bufferSize = getBufferSize(cache.tensorUtil);
    memset(buffer, 0, bufferSize);
    if (_useDmabufIo) {
      _ioTensor->afterWriteToBuffer(t_inputIds->tensor);
    }
  }
  return true;
}

}  // namespace qualla
