#==============================================================================
#
#  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
#  All rights reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
#==============================================================================
from pathlib import Path
import logging

from qti.aisw.core.model_level_api.executor.android_subprocess_executor import AndroidSubprocessExecutor
from qti.aisw.core.model_level_api.model.dlc import DLC

logger = logging.getLogger(__name__)

class OELinuxSubprocessExecutor(AndroidSubprocessExecutor):
    """
    Subprocess Executor class  for OELinux devices, contains pubilc methods to setup an instance,
    run an inference and generate a context_binary
    """

    def __init__(self, device_temp_dir_prefix='/etc/'):
        super().__init__()
        self._device_temp_dir_prefix = device_temp_dir_prefix

    def setup(self, workflow_mode, backend, model, sdk_root, config, output_dir):
        """
        This method is used to load the model and the artifacts to device

        Args:
            workflow_mode: can be WorkflowMode.INFERENCE or WorkflowMode.CONTEXT_BINARY_GENERATOR
            backend: Backend class
            model: Class containing path to the model library or context binary
            sdk_root: Path to SDK
            config: Config file, given to inference

        Returns: None

        Raises:
            ValueError: An error occurred when DLC file path given
            FileNotFoundError: An error occured when a executable or backend lib not found
            RuntimeError: An error occured when wrong workflow_mode given
        """

        if (isinstance(model, DLC)):
            raise ValueError("OELinux Emebedded devices do not support DLC models yet.")
        return super().setup(workflow_mode, backend, model, sdk_root, config, output_dir)

    def run_inference(self, config, backend, model, sdk_root, input_data, output_dir):
        """
        This method loads the artifacts (if not done already) and executes the model with
        the given input data

        Args:
            config: Run config for the infrence step
            backend: Backend class
            model: Class containing the path to the model library or context binary
            sdk_root: Path to SDK
            input_data: Contains the input data as a list of np.array or dict[str, np.ndarray]

            Returns:
                Tuple: A tuple containing:
                    np.ndarray: The Numpy array of the output
                    Profiling_data: An object containing the profiling log and any backend profiling artifacts

            Raises:
                InferenceError: An error occured when execution (netrun) fails
                RuntimeError: An error occured during setup or locating output

        """
        return super().run_inference(config, backend, model, sdk_root, input_data, output_dir)

    def generate_context_binary(self, config, backend, model, sdk_root, output_path, output_filename):
        """
        This method generates the context binary for a given model for a OELinux device

        Args:
            config: Generate config
            backend: Backend class
            model: Class containing the path to the model library or context binary
            sdk_root: Path to SDK
            output_path: Path to store the generated binary
            output_filename: Name of the generated context binary

        Returns:
            Tuple: A tuple containing:
                QnnContextBinary: An object storing the name and path of the generated binary
                Profiling_data: An object containing the profiling log and any backend profiling artifacts
        """
        return super().generate_context_binary(config, backend, model, sdk_root, output_path, output_filename)

    @staticmethod
    def _create_inference_model_argument(model, temp_directory):
        """
        Private method to retrieve the path to the provided model during inference
        """
        try:
            return OELinuxSubprocessExecutor._create_model_lib_argument(model,
                                                                        temp_directory)
        except AttributeError:
            pass

        try:
            return OELinuxSubprocessExecutor._create_context_binary_argument(model,
                                                                             temp_directory)
        except AttributeError:
            raise ValueError("Could not retrieve path to provided model")

    @staticmethod
    def _create_context_binary_generator_model_argument(model, temp_directory):
        """
        Private method to retrieve the path to the provided model during context generation
        """
        try:
            return OELinuxSubprocessExecutor._create_model_lib_argument(model,
                                                                        temp_directory)

        except AttributeError:
            raise ValueError("Could not retrieve path to provided model")

    @staticmethod
    def _get_model_artifacts(model, sdk_root):
        """
        Get a list of path of model artifacts required
        """
        try:
            binary_path = Path(model.binary_path)
            if not binary_path.is_file():
                raise FileNotFoundError(f"Could not find context binary: {binary_path}")
            return [binary_path]
        except AttributeError:
            pass

        try:
            model_path = Path(model.model_path)
            if not model_path.is_file():
                raise FileNotFoundError(f"Could not find model library: {model_path}")
            return [model_path]
        except AttributeError:
            pass