# =============================================================================
#
#  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# =============================================================================

from pathlib import Path
from logging import getLogger
from json import load
from tempfile import TemporaryDirectory
from os import PathLike
import shutil
from typing import Union, Dict, Any, Tuple, Optional, List, NamedTuple

from qti.aisw.tools.core.utilities.devices.utils.subprocess_helper import execute
from qti.aisw.core.model_level_api.utils.qnn_sdk import qnn_sdk_root
from qti.aisw.core.model_level_api.backend.backend import Backend

logger = getLogger(__name__)
default_profiling_log_name = 'qnn-profiling-data_0.log'

class ProfilingData(NamedTuple):
    profiling_log: Path
    backend_profiling_artifacts: Optional[List[Path]]

def profiling_log_to_dict(profiling_log: Union[PathLike, str]) -> Dict[str, Any]:
    """
    A helper function to generate a dictionary from a profiling log.

    Args:
        profiling_log (Union[PathLike, str]): The profiling log to generate a dictionary from

    Returns:
       Dict[str, Any]: A dictionary containing the profiling events reported from the application
                       and backend
    """
    profiling_log = Path(profiling_log)
    if not profiling_log.is_file():
        raise FileNotFoundError("Provided profiling log could not be found")

    sdk_root = qnn_sdk_root()
    qnn_profile_viewer = Path(sdk_root, 'bin', 'x86_64-linux-clang', 'qnn-profile-viewer')
    if not qnn_profile_viewer.is_file():
        raise FileNotFoundError("Could not locate qnn-profile-viewer in QNN SDK")

    json_reader = Path(sdk_root, 'lib', 'x86_64-linux-clang', 'libQnnJsonProfilingReader.so')
    if not json_reader.is_file():
        raise FileNotFoundError("Could not locate libQnnJsonProfilingReader.so in QNN SDK")

    with TemporaryDirectory() as temp_dir:
        output_json = Path(temp_dir, 'profiling_output.json')
        profile_viewer_args = f'--input_log {profiling_log} --output {output_json} --reader {json_reader}'

        logger.debug(f'Running command: {qnn_profile_viewer} {profile_viewer_args}')
        completed_process = execute(qnn_profile_viewer, profile_viewer_args.split())
        if completed_process.returncode != 0:
            raise RuntimeError(f'qnn-profile-viewer execution failed, stdout: '
                               f'{completed_process.stdout}, stderr: {completed_process.stderr}')

        with output_json.open() as f:
            output_dict = load(f)

    return output_dict

def generate_optrace_profiling_output(schematic_bin: Union[PathLike, str],
                                      profiling_log: Union[PathLike, str],
                                      output_dir: Union[PathLike, str] = './output/') \
        -> Tuple[Path, Path]:
    """
    A helper function to generate optrace artifacts (a JSON viewable via chrometrace and an HTML
    summary report) from a profiling log and a schematic bin.

    Args:
        schematic_bin (Union[PathLike, str]): The schematic bin file created during context binary
                                              generation
        profiling_log (Union[PathLike, str]): The profiling log file created during execution
        output_dir (Union[PathLike, str]): The output directory where outputs will be stored

    Returns:
        Tuple[Path, Path]: A tuple containing (path to output JSON, path to HTML summary report)
    """
    schematic_bin = Path(schematic_bin)
    if not schematic_bin.is_file():
        raise FileNotFoundError("Provided schematic bin could not be found")

    profiling_log = Path(profiling_log)
    if not profiling_log.is_file():
        raise FileNotFoundError("Provided profiling log could not be found")

    sdk_root = qnn_sdk_root()
    qnn_profile_viewer = Path(sdk_root, 'bin', 'x86_64-linux-clang', 'qnn-profile-viewer')
    if not qnn_profile_viewer.is_file():
        raise FileNotFoundError("Could not locate qnn-profile-viewer in QNN SDK")

    htp_optrace_reader = Path(sdk_root, 'lib', 'x86_64-linux-clang', 'libQnnHtpOptraceProfilingReader.so')
    if not htp_optrace_reader.is_file():
        raise FileNotFoundError("Could not locate libQnnHtpOptraceProfilingReader.so in QNN SDK")

    output_dir = Path(output_dir).resolve()
    output_dir.mkdir(parents=True, exist_ok=True)

    output_json = output_dir / 'chrometrace.json'
    output_html_summary = output_dir / 'chrometrace_qnn_htp_analysis_summary.html'

    profile_viewer_args = f'--input_log {profiling_log} --output {output_json} ' \
                          f'--schematic {schematic_bin} --reader {htp_optrace_reader}'
    logger.debug(f'Running command: {qnn_profile_viewer} {profile_viewer_args}')
    completed_process = execute(qnn_profile_viewer, profile_viewer_args.split())
    if completed_process.returncode != 0:
        raise RuntimeError(f'qnn-profile-viewer execution failed, stdout: '
                           f'{completed_process.stdout}, stderr: {completed_process.stderr}')

    if not output_json.is_file():
        raise RuntimeError(f'Could not locate optrace json after running qnn-profile-viewer')

    if not output_html_summary.is_file():
        raise RuntimeError(f'Could not locate optrace html summary after running qnn-profile-viewer')

    return output_json, output_html_summary

def move_backend_profiling_artifacts(backend: Backend, output_dir: Union[PathLike, str]) \
    -> Optional[List[Path]]:
    """
    A helper function to query backend profiling artifacts stored in a temp directory and move
    them to a user-specified output directory

    Args:
        backend (Backend): The Backend instance which may have accumulated profiling artifacts
        output_dir (Union[PathLike, str]): The output directory where the artifacts will be stored

    Returns:
        Optional[List[Path]]: A list of profiling artifacts which have been moved into the output
        directory, or None if the Backend instance did not have any accumulated profiling artifacts
    """
    moved_profiling_artifacts = None
    backend_profiling_artifacts = backend.get_profiling_artifacts()
    if backend_profiling_artifacts:
        moved_profiling_artifacts = []
        for artifact in backend_profiling_artifacts:
            shutil.copy(artifact, output_dir)
            moved_profiling_artifacts.append(Path(output_dir, artifact.name))
        backend.clear_profiling_artiacts()

    return moved_profiling_artifacts

def get_backend_profiling_data(backend: Backend, output_dir: Union[PathLike, str],
                               temp_dir: Optional[Union[PathLike, str]] = None) -> ProfilingData:
    """
    A helper function to locate and return a profiling log generated during inference or context
    binary generation. Any backend profiling artifacts will be moved and returned as well.

    Args:
        backend (Backend): The Backend instance that will be queried for backend profiling artifacts
        output_dir (Union[PathLike, str]): The user-provided output directory where the profiling
        log and any backend profiling artifacts should be placed in
        temp_dir (Optional[Union[PathLike, str]]): The temporary directory where the profiling log
        was created in. Should be None if the profiling log is already present in output_dir
    Returns:
         ProfilingData: An object containing the profiling log and any backend profiling artifacts
    """
    output_dir = Path(output_dir)
    output_dir.mkdir(parents=True, exist_ok=True)
    profiling_log_dir = Path(temp_dir) if temp_dir else output_dir
    profiling_log = profiling_log_dir / default_profiling_log_name
    if not profiling_log.is_file():
        raise RuntimeError(f"Could not locate profiling log at {profiling_log}")

    # if the log is in a temp directory instead of the desired output directory, move it
    if temp_dir:
        moved_profiling_log = (output_dir / default_profiling_log_name).resolve()

        # if there is a symlink to a profiling log in the output directory already, the
        # symlink will be followed and the linked file will be overwritten instead of the
        # symlink. To avoid this, remove any existing symlinks before moving the profile log
        # to the output directory.
        if moved_profiling_log.is_symlink():
            moved_profiling_log.unlink()

        shutil.copy(profiling_log, moved_profiling_log)
        profiling_log = moved_profiling_log

    backend_profiling_artifacts = move_backend_profiling_artifacts(backend, output_dir)
    return ProfilingData(profiling_log=profiling_log,
                         backend_profiling_artifacts=backend_profiling_artifacts)
