//=============================================================================
//
//  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
//  All Rights Reserved.
//  Confidential and Proprietary - Qualcomm Technologies, Inc.
//
//=============================================================================

#pragma once

#include <atomic>
#include <memory>
#include <string>

#include "Engine.hpp"
#include "qualla/detail/json.hpp"

using json = qualla::json;

namespace genie {

class Registry {
 public:
  // Static function to get keys from registry
  static std::vector<size_t> getKeysFromRegistry(const json& config);

  // Static function to get engine from registry
  static std::unordered_map<std::string, std::shared_ptr<qualla::Engine>> getEngineFromRegistry(
      const std::vector<size_t>& keys,
      std::shared_ptr<ProfileStat> profileStat,
      std::shared_ptr<genie::log::Logger> logger);

  // Static function to delete engines from registry
  static void deleteEnginesFromRegistry(const std::vector<size_t>& keys);

 private:
  // Helper class to generate a unique key
  class EngineKey {
   public:
    size_t createKey(const json& engineConfig);
    // delete all constructors and destructor
    EngineKey()                            = default;
    ~EngineKey()                           = default;
    EngineKey(const EngineKey&)            = delete;
    EngineKey(EngineKey&&)                 = delete;
    EngineKey& operator=(const EngineKey&) = delete;
    EngineKey& operator=(EngineKey&&)      = delete;
  };

  class RegEngine {
   public:
    RegEngine(const json& engineConfig,
              std::shared_ptr<ProfileStat> profileStat,
              std::shared_ptr<genie::log::Logger> logger);
    ~RegEngine();

    // Delete all other constructors
    RegEngine()                            = delete;
    RegEngine(const RegEngine&)            = delete;
    RegEngine(RegEngine&&)                 = delete;
    RegEngine& operator=(const RegEngine&) = delete;
    RegEngine& operator=(RegEngine&&)      = delete;

    void incrementUseCount();
    void decrementUseCount();
    std::string getRole() const;
    uint32_t getUseCount() const;

    std::shared_ptr<genie::Engine> getGenieEngine() const;

   private:
    std::shared_ptr<genie::Engine> m_engine;
    std::string m_role;
    std::atomic<uint32_t> m_useCount;
  };

  static std::unordered_map<size_t, std::shared_ptr<RegEngine>>& getKeyToEngineMap();
  static std::unordered_map<size_t, json>& getKeyToConfigMap();
  static std::mutex& getMutexRegistryFn();
};

}  // namespace genie