//==============================================================================
//
// Copyright (c) 2020 Qualcomm Technologies, Inc.
// All Rights Reserved.
// Confidential and Proprietary - Qualcomm Technologies, Inc.
//
//==============================================================================

#ifndef SPLITHIST_H
#define SPLITHIST_H

#include <vector>
#include "interface_defs.h"
#include "log.h"

namespace hnnx {
// class to represent a slice history.
// This is just a wrapper around unsigned;
// public methods allow null-construct, copy, compare-eq.
// Also, special 'next_slice' increment operation.
class SplitHistoryTable;

class splithist_t {
    unsigned val;
    explicit splithist_t(unsigned v) : val(v) {}
    friend SplitHistoryTable;
    // this is the number of split which can refer to
    // one record in the table; it determines how many
    // LSBS of 'val' are the slice no; the remainder
    // are table index.
    static unsigned constexpr SPLITPER = 1024;

  public:
    splithist_t() : val(0) {}
    splithist_t(splithist_t const &) = default;
    splithist_t &operator=(splithist_t const &) = default;
    ~splithist_t() = default;

    inline bool empty() const { return val == 0; }
    // the following 2 methods only work if nsplit < SPLITPER
    // they are only intended for results of AUTOTHREAD
    unsigned get_record() const { return val / SPLITPER; }
    unsigned get_sliceno() const { return val % SPLITPER; }

    inline bool operator==(splithist_t const &rhs) const { return val == rhs.val; }
    inline bool operator!=(splithist_t const &rhs) const { return val != rhs.val; }
    inline void next_slice()
    {
        unsigned nextval = val + 1;
        if ((nextval & (SPLITPER - 1)) == 0) { // rolled over
            nextval -= SPLITPER * 2; // reset, and back one record.
        }
        val = nextval;
    }
    inline unsigned value() const { return val; }
};
class SplitHistory {
    friend SplitHistoryTable;

  public:
    unsigned orig_id = 0;
    unsigned unique_id = 0;
    typedef std::vector<std::tuple<int, int, int>> indices_list_t;
    indices_list_t split_indices;
    SplitHistory() : orig_id(0), unique_id(0), split_indices() {}
    SplitHistory(SplitHistory const &) = default;
    SplitHistory(SplitHistory &&) = default;
    SplitHistory &operator=(SplitHistory const &) = default;
    SplitHistory &operator=(SplitHistory &&) = default;
    ~SplitHistory() = default;

    bool empty() const { return split_indices.empty(); }
    std::string indices_string() const;
    std::string indices_string_raw() const;
    std::vector<int> offsets() const;

  private:
    SplitHistory(unsigned orig_id_in, unsigned unique_id_in, indices_list_t &&spl_in)
        : orig_id(orig_id_in), unique_id(unique_id_in), split_indices(std::move(spl_in))
    {
    }
};

// The table in SplitHistoryTable represents the history of 'node split events' for any
// given OpDef, with a series of trees encoded in a table. Nodes with the same history
// have the same 32-bit splithist_t (and references are not counted).
// For each slice generated by an autosplit, we represent the dimension sliced and the slice
// index; if the op slices was not previously sliced, we retain its 'original id' in the root
// of a tree; if it was previously sliced, we retain the previous split information as a reference
// towards the root of the tree.
// Since all slices within a split have the same information other than the slice index, we
// store that in the 'pointer': the splithist_t (in its upper bits) references one of the entries
// in the table via a 1-based index; and the lower bits of splithist_t are a slice index. So, only
// one record needs to be added to the tree for each AUTOSPLIT done:
//
// - If a OpDef Q which was never split before is split on a given index, we make a new
//   'root' record containing the original id from Q, and the dimension. The splithist_t generated
//   for each of the parts all have the same table index - to the new record - but the
//   'slice index' are all different : 0,1,2 ...
// - If an OpDef Q was previously split (and thus has  non-zero splithist), we instead
//   make a 'subsplit' record, which contains the splithist_t of Q (this acts as a
//   pointer towards the tree root), and the split dim.
//
//  If any split involves more than SPLITPER parts, the slice indices won't all fit in the
//  the lower bits of splithist_t. To support this, we add 'extension' records ahead of
//  the normal records in the table.
//  E.g. if SPLITPER=1024 and we need to make 2500 split, we need two extension records,
//  ahead of the 'root' or 'subsplit' entry, as follows in the table:
//
//         - an extension record with recoff = 2
//         - an extension record with recoff = 1
//         - the main record (root or subsplit).
//
//   Now, for the first 1024 slices of that split [0..1023], the splithist_t upper bits contain the index
//   of the main record, and have slice_idx 0..1023; the next 1024 slices [1024..2047] have upper bits
//   referencing the extension record with recoff=1, also with slice_idx = 0..0123; and the final 452 slices
//   [2048..2499] have upper bits referencing the extension record with recoff=2, and have slice_idx = 0..451.
//   Extension records occupy one entry in the table, so when an extension record is encountered when following
//   a splithist_t, it is easy to locate the 'main' record it refers to, and to adjust the slice index to the proper
//   value, by using the 'recoff' in the extension record.
//
// So, there are three types of records:
//   - Root record, contains a 3-bit 'dimno', and 32-bit 'original op id' (the lower 32 bits of the OpId which is split)
//   - Subsplit record, contains a 3-bit dimno, and 'subsplit_t' which points one level towards the root (and, it may point to
//     an extension record, if the containing split has more than SPLITPER slices).
//   - extension records, which contain a 'recoff' value.
// These records are stored in a vector of 'unsigned'; in addition to the fields given above, there
// are upper bits in the first word which signify what the record type is. Also, the records are often packed
// into one word, sometimes two, according to the magnitude of the values to be encoded. The 'extension' record always
// fits in one word.
// The details of the encoding are described in splithist.cc, near the three methods of SplitHistoryTable which
// need to know about them (e.g. cursor_from_splithist unpacks a record to 'Cursor' object, compensating
// for extension record when present).

class SplitHistoryTable {
    std::vector<unsigned> table;
    static unsigned constexpr SPLITPER = splithist_t::SPLITPER;
    // MAX_SLICE_N determines the max # of splits which can be done at any one
    // time, which limits the number of extension words allowed. This is set
    // to a very high value here; it's really just to provide an upper limit
    // for sanity-checking the data structure, and should not be the deciding
    // limit (max # of inputs on a concat).
    static unsigned constexpr MAX_SLICE_N = 4 * 1024 * 1024;
    static unsigned constexpr MAX_EXTRECS = (MAX_SLICE_N - 1) / SPLITPER;

  public:
    SplitHistoryTable() = default;
    splithist_t make_new_split(uint32_t dimno, OpId nodeid, splithist_t oldhist, int nslices, uint32_t slice_size,
                               uint32_t first_sliceno = 0);
    SplitHistory get_splithist(splithist_t shist) const;
    // Equivalent to get_splithist, but it only returns {orig_id, unique_id}
    // instead of the full SplitHistory, and will be faster as a result.
    std::pair<unsigned, unsigned> get_splithist_ids(splithist_t shist) const;

    // Return the main split record / slice number.
    // This differs from the values returned by splithist_t get_record() / get_sliceno() for extension records.
    unsigned get_splithist_main_record(splithist_t shist) const;
    unsigned get_splithist_main_sliceno(splithist_t shist) const;

    // When two nodes containing splithist_t values 'a' and 'b' are combined
    // by CSE, this determines the splithist for the combined node.
    splithist_t resolve(splithist_t const a, splithist_t const b, const bool is_const) const
    {
        return ((a == b) || b.empty()) ? a : a.empty() ? b : resolve_func(a, b, is_const);
    }

    // Return true iff splithist_b is empty or is a parent of splithist_a
    bool is_parent_of(splithist_t a, splithist_t b) const;
    // Return the immediate parent of a
    splithist_t get_parent(splithist_t a) const;

  protected:
    splithist_t resolve_func(splithist_t a, splithist_t b, bool is_const) const;

    struct Cursor {
        unsigned rec_index; // current record's index (skipping indirect record if any)
        splithist_t parent; // link to parent, or empty (0) if this is root.
        int sliceno; // slice index (adjusted for indirect record if any)
        int dimno;
        int slicesize;
        inline bool is_root() const { return parent.empty(); }
    };
    // internal methods to traverse from a position to the root.
    void cursor_from_splithist(Cursor &, splithist_t const &) const;
    void cursor_to_parent(Cursor &c) const; // move to parent, only legal if not root.
    unsigned extract_orig_id(unsigned recindex) const; // only to be used on root record.
};

} // end namespace hnnx

#endif // SPLITHIST_H
