# ==============================================================================
#
# Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
# All Rights Reserved.
# Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# ==============================================================================
"""
GenAI Builder constructed to shepherd a model from source framework to being prepared to run.
GenAI Builder builds a GenAIContainer object with all of the prepared model artifacts.
"""

import os
from abc import ABC, abstractmethod
from os import PathLike
from pathlib import Path

from transformers.configuration_utils import PretrainedConfig

from qairt.api.configs.common import BackendType
from qairt.gen_ai_api.configs.gen_ai_config import GenAIConfig
from qairt.gen_ai_api.containers.gen_ai_container import GenAIContainer
from qairt.modules.genie_execution.genie_config import PositionalEncoding, PositionalEncodingType
from qairt.utils.loggers import get_logger

logger = get_logger(__name__)


class GenAIBuilder(ABC):
    """
    Abstract class for Generative AI Builder.
    """

    def __init__(self, framework_model_path: PathLike | str, config: GenAIConfig, backend: BackendType):
        self.framework_model_path = Path(framework_model_path).resolve()
        self.config = config
        self._backend: BackendType = backend

    @classmethod
    def _create_config_from_pretrained(cls, config: PretrainedConfig) -> GenAIConfig:
        """
        Creates a GenAI configuration from a pretrained model configuration.

        Args:
            config: The config object from the Hugging Face transformers library.

        Returns:
            GenAI configuration object.
        """
        tokenizer_path = os.path.join(config._name_or_path, "tokenizer.json")
        n_embd = getattr(config, "hidden_size", None)
        n_heads = getattr(config, "num_attention_heads", None)
        gen_ai_config = GenAIConfig(
            tokenizer_path=tokenizer_path,
            context_length=4096,
            n_vocab=getattr(config, "n_vocab", getattr(config, "vocab_size")),
            n_heads=n_heads,
            n_layer=getattr(config, "num_hidden_layers", None),
            n_embd=n_embd,
            bos_token=config.bos_token_id,
            eos_token=config.eos_token_id,
            eot_token=getattr(config, "eot_token_id", None),
            rope_theta=getattr(config, "rope_theta", None),
        )
        return gen_ai_config

    @abstractmethod
    def build(self) -> GenAIContainer:
        """
        Prepares and builds the model for execution.

        Returns:
            GenAIContainer object.
        """
        raise NotImplementedError("build must be defined")
