# =============================================================================
#
# Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
# All rights reserved.
# Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# =============================================================================
from pathlib import Path
from typing import Any, Optional

from qti.aisw.accuracy_debugger.encodings.encodings_utils import (
    EncodingVersion,
    TensorDtype,
    TensorType,
    UnsupportedEncodingsVersionError,
    get_encodings_version,
)
from qti.aisw.accuracy_debugger.utils.file_utils import dump_json, read_json
from qti.aisw.converters.common import ir_graph
from qti.aisw.dlc_utils import modeltools


# ------------------------------------------------------
# Code block for Tensor encoding
# ------------------------------------------------------


class TensorEncoding:
    """Base class for storing encoding information for an tensor"""

    def __init__(self, tensor_name: str, tensor_type: TensorType) -> None:
        """Initializes TensorEncoding class"""
        self.tensor_name: str = tensor_name
        self.tensor_type: TensorType = tensor_type
        self.dtype: TensorDtype = None
        self.bitwidth: int = None
        self.min: list = []
        self.max: list = []
        self.scale: list = []
        self.offset: list = []
        self.is_symm: str = ""
        self.channels: int = None
        self.axis: int = None

    @property
    def V0(self) -> list[dict[str, Any]]:
        """Version 0.6.0 representation of an tensor encoding"""
        # Can not create V0 encoding representation if tensor is of type TensorType.Encodings
        if self.tensor_type == TensorType.Encodings:
            raise ValueError(
                f"Tensor of type: {TensorType.Encodings} can not be represented in"
                f"{EncodingVersion.V0.value} version."
            )

        encoding_repr = []

        # If encoding is of type float
        if self.dtype == TensorDtype.FLOAT:
            encoding_repr = [{"dtype": self.dtype.value, "bitwidth": self.bitwidth}]
        else:
            for channel_index in range(self.channels):
                channel_encoding = {
                    "bitwidth": self.bitwidth,
                    "is_symmetric": self.is_symm,
                    "max": self.max[channel_index],
                    "min": self.min[channel_index],
                    "offset": self.offset[channel_index],
                    "scale": self.scale[channel_index],
                }
                encoding_repr.append(channel_encoding)

        return encoding_repr

    @property
    def V1(self) -> dict[str, Any]:
        """Version 1.0.0 representation of an tensor encoding"""
        # Can not create V1 encoding representation if tensor is of type TensorType.Encodings
        if self.tensor_type == TensorType.Encodings:
            raise ValueError(
                f"Tensor of type: {TensorType.Encodings} can not be represented in"
                f"{EncodingVersion.V1.value} version."
            )

        # If encoding is of type float
        if self.dtype == TensorDtype.FLOAT:
            encoding_repr = {
                "dtype": self.dtype.value.upper(),
                "bw": self.bitwidth,
                "name": self.tensor_name,
            }
        else:
            encoding_repr = {
                "bw": self.bitwidth,
                "dtype": "INT",
                "enc_type": "PER_TENSOR" if self.channels == 1 else "PER_CHANNEL",
                "is_sym": self.is_symm,
                "name": self.tensor_name,
                "offset": self.offset,
                "scale": self.scale,
            }

        return encoding_repr

    @property
    def V2(self) -> dict[str, Any]:
        """Version 2.0.0 representation of an tensor encoding"""
        # Channel Wise or Row Wise TensorEncoding created with V0 or V1 encoding json file can not
        # be converted to V2 encoding version because axis information is not present
        if self.channels and self.channels > 1 and (self.axis is None):
            raise Exception(
                "Per Channel/Row TensorEncoding created with V1 or V0 encoding json file can not be"
                "converted to V2 encoding version"
            )

        # If encoding is of type float
        if self.dtype == TensorDtype.FLOAT:
            encoding_repr = {
                "output_dtype": f"{self.dtype.value}{self.bitwidth}",
                "name": self.tensor_name,
            }

        else:
            encoding_repr = {
                "output_dtype": f"{self.dtype.value}{self.bitwidth}",
                "name": self.tensor_name,
            }
            # Following scenarios
            # 1. Per tensor
            # 2. Per Channel
            if self.channels == 1:
                encoding_repr["y_scale"] = self.scale[0]
                if self.offset[0] != 0:
                    encoding_repr["y_zero_point"] = self.offset[0]
            else:
                encoding_repr["y_scale"] = self.scale
                encoding_repr["axis"] = self.axis
                # If all offsets are 0, skip it. Not requried.
                if not all(offset == 0 for offset in self.offset):
                    encoding_repr["y_zero_point"] = self.offset

        return encoding_repr


class DlcTensorEncoding(TensorEncoding):
    """Encoding class for a Dlc Tensor"""

    def __init__(self, ir_tensor: Any) -> None:
        """Initializes DlcTensorEncoding class

        Args:
            ir_tensor: Object of libPyIrGraph.IrStaticTensor or libPyIrGraph.IrTensor.
        """
        tensor_name = ir_tensor.name()
        tensor_type = (
            TensorType("param_encodings")
            if ir_tensor.is_static_tensor()
            else TensorType("activation_encodings")
        )
        super().__init__(tensor_name=tensor_name, tensor_type=tensor_type)

        # Check if the tensor is running in float or int
        float_types = [ir_graph.QNN_DATATYPE_FLOAT_16, ir_graph.QNN_DATATYPE_FLOAT_32]
        if ir_tensor.data_type() in float_types:
            self._set_float_encodings(ir_tensor=ir_tensor)
        else:
            self._set_quantized_encodings(ir_tensor=ir_tensor)

    def _consume_qnn_quantization_encoding_scale_offset(self, encInfo: Any) -> None:
        """Given a Per Tensor Encoding info, extract the info

        Args:
            encInfo: Object of Qnn_QuantizationEncoding_t.QNN_QUANTIZATION_ENCODING_SCALE_OFFSET
        """
        self.bitwidth = encInfo.bw
        self.min = [encInfo.min]
        self.max = [encInfo.max]
        self.scale = [encInfo.scale]
        self.offset = [encInfo.offset]
        self.is_symm = str(encInfo.is_symmetric).lower()
        self.channels = 1

    def _consume_qnn_quantization_encoding_axis_scale_offset(self, axisEncInfo: Any) -> None:
        """Given a Per Channel Encoding info, extract the info

        Args:
            axisEncInfo: Object of Qnn_QuantizationEncoding_t.QNN_QUANTIZATION_ENCODING_AXIS_SCALE_OFFSET
        """
        encInfos = axisEncInfo.encInfos
        self.bitwidth = encInfos[0].bw
        self.is_symm = str(encInfos[0].is_symmetric).lower()
        self.channels = len(encInfos)

        for encInfo in encInfos:
            self.min.append(encInfo.min)
            self.max.append(encInfo.max)
            self.scale.append(encInfo.scale)
            self.offset.append(encInfo.offset)

    def _set_quantized_encodings(self, ir_tensor: Any) -> None:
        """Given ir_tensor extract the quantization info for the tensor

        Args:
            ir_tensor: Object of libPyIrGraph.IrStaticTensor or libPyIrGraph.IrTensor.

        Raises:
            Exception: If quantized tensor encoding type is not one of the following:
                1. ir_graph.QNN_QUANTIZATION_ENCODING_SCALE_OFFSET
                2. ir_graph.QNN_QUANTIZATION_ENCODING_AXIS_SCALE_OFFSET
        """
        int_type = [
            ir_graph.QNN_DATATYPE_SFIXED_POINT_8,
            ir_graph.QNN_DATATYPE_SFIXED_POINT_16,
            ir_graph.QNN_DATATYPE_SFIXED_POINT_32,
        ]
        if ir_tensor.data_type() in int_type:
            self.dtype = TensorDtype.SFXP
        else:
            self.dtype = TensorDtype.UFXP

        tensor_encoding = ir_tensor.get_encoding()
        if tensor_encoding.type == ir_graph.QNN_QUANTIZATION_ENCODING_SCALE_OFFSET:
            self._consume_qnn_quantization_encoding_scale_offset(encInfo=tensor_encoding.encInfo)
        elif tensor_encoding.type == ir_graph.QNN_QUANTIZATION_ENCODING_AXIS_SCALE_OFFSET:
            self.axis = (
                tensor_encoding.axisEncInfo.axis
            )  # This axis value is not same as framework axis.
            self._consume_qnn_quantization_encoding_axis_scale_offset(
                axisEncInfo=tensor_encoding.axisEncInfo
            )
        else:
            raise Exception(
                f"Tensor encoding of type {tensor_encoding.type.name} is not supported."
            )

    def _set_float_encodings(self, ir_tensor: Any) -> None:
        """Given ir_tensor extract the float info for the tensor

        Args:
            ir_tensor: Object of libPyIrGraph.IrStaticTensor or libPyIrGraph.IrTensor.
        """
        self.dtype = TensorDtype.FLOAT
        if ir_tensor.data_type() == ir_graph.QNN_DATATYPE_FLOAT_16:
            self.bitwidth = 16
        else:
            self.bitwidth = 32


class V0JsonTensorEncoding(TensorEncoding):
    """Encoding class for a Version 0.6.0 tensor encoding loaded from json file"""

    def __init__(self, tensor_name: str, tensor_encoding: list, tensor_type: TensorType) -> None:
        """Initializes V0JsonTensorEncoding class

        Args:
            tensor_name: Name of the tensor.
            tensor_encoding: Version "0.6.0" tensor encoding.
            tensor_type: Type of the tensor: [param or activation]
        """
        super().__init__(tensor_name=tensor_name, tensor_type=tensor_type)
        self._tensor_encoding = tensor_encoding
        self.dtype = self._get_dtype()
        self.bitwidth = self._tensor_encoding[0]["bitwidth"]
        self.scale = self._get_value(field="scale")
        self.offset = self._get_value(field="offset")
        self.min = self._get_value(field="min")
        self.max = self._get_value(field="max")
        self.is_symm = str(self._tensor_encoding[0].get("is_symmetric", "")).lower()
        self.channels = len(self.scale)

    def _get_dtype(self) -> TensorDtype:
        """Extracts dtype from V0 tensor encoding

        Returns:
            (TensorDtype): dtype of the tensor encoding
        """
        dtype = TensorDtype.FLOAT
        if "dtype" in self._tensor_encoding[0]:
            dtype = TensorDtype(self._tensor_encoding[0]["dtype"].lower())
        elif "scale" in self._tensor_encoding[0]:
            if self._tensor_encoding[0]["scale"] == 0:
                dtype = TensorDtype.FLOAT
            else:
                dtype = TensorDtype.SFXP

        return dtype

    def _get_value(self, field: str) -> list:
        """Extarct the scale or offset from the V0 tensor encoding

        Args:
            field (str): either scale or offset

        Returns:
            (list): list of field values
        """
        value = []

        if field in self._tensor_encoding[0]:
            for channel in self._tensor_encoding:
                value.append(channel[field])

        return value


class V1JsonTensorEncoding(TensorEncoding):
    """Encoding class for a Version 1.0.0 tensor encoding loaded from json file"""

    def __init__(self, tensor_encoding: dict, tensor_type: TensorType) -> None:
        """Initializes V1JsonTensorEncoding class

        Args:
            tensor_encoding: Version "1.0.0" tensor encoding.
            tensor_type: Type of the tensor: [param or activation]
        """
        super().__init__(tensor_name=tensor_encoding["name"], tensor_type=tensor_type)
        self._tensor_encoding = tensor_encoding
        self.dtype = TensorDtype(self._tensor_encoding["dtype"].lower())
        self.bitwidth = self._tensor_encoding["bw"]
        self.scale = self._tensor_encoding.get("scale", [])
        self.offset = self._tensor_encoding.get("offset", [])
        self.is_symm = str(self._tensor_encoding.get("is_sym", "")).lower()
        self.channels = len(self.scale)
        if self.dtype in [TensorDtype.SFXP, TensorDtype.UFXP]:
            self.min = [offset * scale for offset, scale in zip(self.offset, self.scale)]
            self.max = [
                (2 ** (self.bitwidth) - 1) * scale + min for scale, min in zip(self.scale, self.min)
            ]


class V2JsonTensorEncoding(TensorEncoding):
    """Encoding class for a Version 2.0.0 tensor encoding loaded from json file"""

    def __init__(self, tensor_encoding: dict) -> None:
        """Initializes V2JsonTensorEncoding class

        Args:
            tensor_encoding: Version 2.0.0 tensor encoding
        """
        super().__init__(tensor_name=tensor_encoding["name"], tensor_type=TensorType("encodings"))
        # AIMET: Positive offset value
        # QNN: Negative offset value
        # Since Qairt never dumps in V2 format, we assume this behavior will remain going ahead.
        # Hence, we multiply offset with -1 assuming this class will be used only by AIMET enc file

        self.scale = tensor_encoding.get("y_scale", [])
        self.channels = len(self.scale)
        offset = tensor_encoding.get("y_zero_point", [0] * self.channels)
        self.offset = [-1 * o for o in offset]  # Convert AIMET offset to QNN format

        if "float" in tensor_encoding["output_dtype"]:
            self.dtype = TensorDtype.FLOAT
            self.bitwidth = int(tensor_encoding["output_dtype"].split("float")[1])
        elif "uint" in tensor_encoding["output_dtype"]:
            self.dtype = TensorDtype.UFXP
            self.bitwidth = int(tensor_encoding["output_dtype"].split("uint")[1])
            if self.offset[0] == -1 * 2 ** (self.bitwidth - 1):
                self.is_symm = "true"
            else:
                self.is_symm = "false"
            min_point = 0  # E.g. 0 for int8
            max_point = pow(2, self.bitwidth) - 1  # E.g. 255 for int8
        else:
            self.dtype = TensorDtype.SFXP
            self.bitwidth = int(tensor_encoding["output_dtype"].split("int")[1])
            if self.offset[0] == 0:
                self.is_symm = "true"
            else:
                self.is_symm = "false"
            min_point = -1 * pow(2, self.bitwidth - 1)  # E.g. -128 for int8
            max_point = pow(2, self.bitwidth - 1) - 1  # E.g. 127 for int8

        for channel_scale, channel_offset in zip(self.scale, self.offset):
            self.min.append((min_point + channel_offset) * channel_scale)
            self.max.append((max_point + channel_offset) * channel_scale)

        self.axis = tensor_encoding.get("axis", None)


# ------------------------------------------------------
# Code block for base Encoding class
# ------------------------------------------------------


class Encoding:
    """Encoding Base Class"""

    def __init__(self) -> None:
        """Initializes Encoding class"""
        self.tensor_encodings: dict[str:TensorEncoding] = {}
        self.name: str = None

    @staticmethod
    def get_encodings_structure(version: EncodingVersion) -> dict:
        """Given encodings version returns empty encodings data structure

        Args:
            version (EncodingVersion): encodings version enum

        Returns:
            (dict or None) encodings data structure

        Raises:
            UnsupportedEncodingsVersionError: If encodings version is not supported
        """
        if version == EncodingVersion.V0:
            return {"activation_encodings": {}, "param_encodings": {}}
        elif version == EncodingVersion.V1:
            return {"version": version.value, "activation_encodings": [], "param_encodings": []}
        elif version == EncodingVersion.V2:
            return {"version": version.value, "encodings": []}

        raise UnsupportedEncodingsVersionError(
            f"Could not create encodings data structure for the given {version} version"
        )

    def _generate_V0_encoding(self) -> dict:
        """Generates an encoding dictionary with list of tensor_encodings in V0 format."""
        encoding = Encoding.get_encodings_structure(version=EncodingVersion.V0)
        for tensor_encoding in self.tensor_encodings.values():
            encoding[tensor_encoding.tensor_type.value][tensor_encoding.tensor_name] = (
                tensor_encoding.V0
            )

        return encoding

    def _generate_V1_encoding(self) -> dict:
        """Generates an encoding dictionary with list of tensor_encodings in V1 format."""
        encoding = Encoding.get_encodings_structure(version=EncodingVersion.V1)
        for tensor_encoding in self.tensor_encodings.values():
            encoding[tensor_encoding.tensor_type.value].append(tensor_encoding.V1)

        return encoding

    def _generate_V2_encoding(self) -> dict:
        """Generates an encoding dictionary with list of tensor_encodings in V2 format."""
        encoding = Encoding.get_encodings_structure(version=EncodingVersion.V2)
        for tensor_encoding in self.tensor_encodings.values():
            encoding[TensorType.Encodings.value].append(tensor_encoding.V2)

        return encoding

    def get_type_encodings(self, tensor_type: TensorType) -> dict[str:TensorEncoding]:
        """Return the dict of all tensor encodings of the given tensor_type with name as
        keys and TensorEncoding object as value.

        Args:
            tensor_type: TensorType Enum to indicate which tensor encodings is needed.
        """
        tensor_encodings = {}

        if tensor_type == TensorType.ActivationEncodings:
            for tensor_name, tensor_encoding in self.tensor_encodings.items():
                if tensor_encoding.tensor_type == TensorType.ActivationEncodings:
                    tensor_encodings[tensor_name] = tensor_encoding
        elif tensor_type == TensorType.ParamEncodings:
            for tensor_name, tensor_encoding in self.tensor_encodings.items():
                if tensor_encoding.tensor_type == TensorType.ParamEncodings:
                    tensor_encodings[tensor_name] = tensor_encoding
        elif tensor_type == TensorType.Encodings:
            tensor_encodings = self.tensor_encodings
        else:
            raise ValueError(f"Incorrect tensor_type: {tensor_type}")

        return tensor_encodings

    def get_tensor_encoding(self, tensor_name: str) -> TensorEncoding:
        """Given tensor name return the object of TensorEncoding

        Args:
            tensor_name: Tensor name for which encoding object is needed.

        Returns:
            (TensorEncoding): Object of TensorEncoding containg the quant info for the tensor.
            (None): If tensor_name not present in the present encoding list
        """
        return self.tensor_encodings.get(tensor_name, None)

    def encoding(self, version: EncodingVersion) -> dict:
        """Generates encoding dictionary with the list of tensor encodings.

        Args:
            version: Encoding Version in which encoding dictionary needs to be generated.

        Returns:
            (dict): encoding dictionary in the provided version.

        Raises:
            (UnsupportedEncodingsVersionError): If the provided version is not supported.
        """
        if version == EncodingVersion.V0:
            return self._generate_V0_encoding()
        elif version == EncodingVersion.V1:
            return self._generate_V1_encoding()
        elif version == EncodingVersion.V2:
            return self._generate_V2_encoding()
        else:
            raise UnsupportedEncodingsVersionError(f"Unsupported encoding {version}")

    def dump(self, version: EncodingVersion, file_path: Path | str) -> None:
        """Dumps the graph encoding consisting of list of tensor encodings according to the
        provided encoding version.

        Args:
            version: EncodingVersion in which encodings should be dumped.
            file_path: path to which encoding file should be dumped.
        """
        encoding = self.encoding(version=version)
        dump_json(data=encoding, json_path=file_path)

    def add(
        self,
        tensor_encoding: Optional[TensorEncoding] = None,
        subgraph_tensor_encodings: Optional[dict[TensorEncoding]] = None,
        overwrite_existing: bool = True,
    ) -> None:
        """Adds the given tensor encoding or a dictionary of subgraph tensor encodings to
        list of graph tensor encodings.

        Args:
            tensor_encoding: Object of TensorEncoding representing an tensor encoding.
            subgraph_tensor_encodings: Dictionary of subgraph tensor encodings with tensor names as
                keys and TensorEncoding object as value.
            overwrite_existing: If True, overwrites the existing tensor encoding with given the
                tensor encoding for a tensor name.

        Raises:
            ValueError: If both tensor_encoding and subgraph_tensor_encodings are None.
        """
        if not (tensor_encoding or subgraph_tensor_encodings):
            raise ValueError(
                "One of tensor_encoding or subgraph_tensor_encodings must be provided."
            )
        if tensor_encoding and (
            tensor_encoding.tensor_name not in self.tensor_encodings or overwrite_existing
        ):
            self.tensor_encodings[tensor_encoding.tensor_name] = tensor_encoding

        if subgraph_tensor_encodings:
            for tensor_name, tensor_encoding in subgraph_tensor_encodings.items():
                if tensor_encoding.tensor_name not in self.tensor_encodings or overwrite_existing:
                    self.tensor_encodings[tensor_name] = tensor_encoding


# ------------------------------------------------------
# Code block for full model encodings
# ------------------------------------------------------


class ModelEncoding(Encoding):
    """Class for model encodings"""

    def __init__(self) -> None:
        """Initializes ModelEncoding class"""
        super().__init__()
        self.activation_tensors: set[str] = set()
        self.param_tensors: str[str] = set()

    def load(self, artifact: str | Path, load_dlc: bool = False, load_json: bool = False) -> None:
        """Loads the model encodings from the given artifact.

        Args:
            artifact: Path to a quantized dlc file or encoding json file.
            load_dlc: Pass True if the artifact provided is dlc file.
            load_json: Pass True if the artifact provided is encoding json file.

        Raises:
            ValueError: if both load_dlc and load_json are True
        """
        self.name = Path(artifact).name.split(".")[0]

        if load_dlc and load_json:
            raise ValueError("A artifact can not be both dlc and encoding file at the same time.")
        elif load_dlc:
            self.param_tensors, self.activation_tensors = self._load_from_dlc(
                quantized_dlc_path=artifact
            )
        elif load_json:
            self._load_from_json(encoding_json=read_json(artifact))
        else:
            raise ValueError("Either of load_dlc or load_json should be True.")

    def _load_v0_json(self, encoding_json: dict) -> None:
        """Loads model encodings from version 0.6.0 encoding file.

        Args:
            encoding_json: dictionary of 0.6.0 version model encoding json.
        """
        for encoding_type in encoding_json:
            if encoding_type in ["activation_encodings", "param_encodings"]:
                tensor_type = TensorType(encoding_type)
                for tensor_name, tensor_encoding in encoding_json[encoding_type].items():
                    self.tensor_encodings[tensor_name] = V0JsonTensorEncoding(
                        tensor_name=tensor_name,
                        tensor_encoding=tensor_encoding,
                        tensor_type=tensor_type,
                    )

    def _load_v1_json(self, encoding_json: dict) -> None:
        """Loads model encodings from version 1.0.0 encoding file.

        Args:
            encoding_json: dictionary of 1.0.0 version model encoding json.
        """
        for encoding_type in encoding_json:
            if encoding_type in ["activation_encodings", "param_encodings"]:
                tensor_type = TensorType(encoding_type)
                for tensor_encoding in encoding_json[encoding_type]:
                    tensor_name = tensor_encoding["name"]
                    self.tensor_encodings[tensor_name] = V1JsonTensorEncoding(
                        tensor_encoding=tensor_encoding, tensor_type=tensor_type
                    )

    def _load_v2_json(self, encoding_json: dict) -> None:
        """Loads model encodings from version 2.0.0 encoding file.

        Args:
            encoding_json: dictionary of 2.0.0 version model encoding json.
        """
        for tensor_encoding in encoding_json["encodings"]:
            self.tensor_encodings[tensor_encoding["name"]] = V2JsonTensorEncoding(
                tensor_encoding=tensor_encoding
            )

    def _load_from_json(self, encoding_json: dict) -> None:
        """Loads model encodings from an encoding file.

        Args:
            encoding_json: dictionary model encoding json.
        """
        json_version = get_encodings_version(encodings=encoding_json)

        if json_version == EncodingVersion.V0:
            self._load_v0_json(encoding_json=encoding_json)
        elif json_version == EncodingVersion.V1:
            self._load_v1_json(encoding_json=encoding_json)
        elif json_version == EncodingVersion.V2:
            self._load_v2_json(encoding_json=encoding_json)

    def _load_from_dlc(self, quantized_dlc_path: Path | str) -> tuple[set, set]:
        """Loads model encodings from quantized dlc.

        Args:
            quantized_dlc_path: Path to the quantized dlc file.

        Returns: tuple of:
            1. set: set of param tensor names
            2. set: set of activation tensor names
        """
        param_tensors = set()
        activation_tensors = set()
        model_reader = modeltools.IrDlcReader()
        model_reader.open(quantized_dlc_path)
        ir_graph = model_reader.get_ir_graph()
        tensor_map = ir_graph.get_tensor_map()

        for tensor_name, ir_tensor in tensor_map.items():
            # Create DlcTensorEncoding object iff tensor is quantizable. This eliminates tensors
            # like int64, int32, boolean etc
            if ir_tensor.is_quantizable():
                self.tensor_encodings[tensor_name] = DlcTensorEncoding(ir_tensor=ir_tensor)
                if ir_tensor.is_static_tensor():
                    param_tensors.update([ir_tensor.name()])
                else:
                    activation_tensors.update([ir_tensor.name()])

        return param_tensors, activation_tensors


# ------------------------------------------------------
# Code block for Subgraph encoding
# ------------------------------------------------------


class SubgraphEncoding(Encoding):
    """Class for Subgraph encodings"""

    def __init__(self):
        """Initializes SubgraphEncoding class"""
        super().__init__()
