##############################################################################
# MIT License
#
# Copyright (c) 2021 OpenAI
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#
# Source: https://github.com/openai/CLIP/blob/main/notebooks/Prompt_Engineering_for_ImageNet.ipynb
# License: https://github.com/openai/CLIP/blob/main/LICENSE
#
##############################################################################
# =============================================================================
#
# Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
# All rights reserved.
# Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# =============================================================================
import math

import numpy as np
from qti.aisw.tools.core.utilities.data_processing import ImageRepresentation, PreProcessor
from qti.aisw.tools.core.utilities.data_processing.utils import Helper


IMAGENET_CLASSES = [
    "tench",
    "goldfish",
    "great white shark",
    "tiger shark",
    "hammerhead shark",
    "electric ray",
    "stingray",
    "rooster",
    "hen",
    "ostrich",
    "brambling",
    "goldfinch",
    "house finch",
    "junco",
    "indigo bunting",
    "American robin",
    "bulbul",
    "jay",
    "magpie",
    "chickadee",
    "American dipper",
    "kite (bird of prey)",
    "bald eagle",
    "vulture",
    "great grey owl",
    "fire salamander",
    "smooth newt",
    "newt",
    "spotted salamander",
    "axolotl",
    "American bullfrog",
    "tree frog",
    "tailed frog",
    "loggerhead sea turtle",
    "leatherback sea turtle",
    "mud turtle",
    "terrapin",
    "box turtle",
    "banded gecko",
    "green iguana",
    "Carolina anole",
    "desert grassland whiptail lizard",
    "agama",
    "frilled-necked lizard",
    "alligator lizard",
    "Gila monster",
    "European green lizard",
    "chameleon",
    "Komodo dragon",
    "Nile crocodile",
    "American alligator",
    "triceratops",
    "worm snake",
    "ring-necked snake",
    "eastern hog-nosed snake",
    "smooth green snake",
    "kingsnake",
    "garter snake",
    "water snake",
    "vine snake",
    "night snake",
    "boa constrictor",
    "African rock python",
    "Indian cobra",
    "green mamba",
    "sea snake",
    "Saharan horned viper",
    "eastern diamondback rattlesnake",
    "sidewinder rattlesnake",
    "trilobite",
    "harvestman",
    "scorpion",
    "yellow garden spider",
    "barn spider",
    "European garden spider",
    "southern black widow",
    "tarantula",
    "wolf spider",
    "tick",
    "centipede",
    "black grouse",
    "ptarmigan",
    "ruffed grouse",
    "prairie grouse",
    "peafowl",
    "quail",
    "partridge",
    "african grey parrot",
    "macaw",
    "sulphur-crested cockatoo",
    "lorikeet",
    "coucal",
    "bee eater",
    "hornbill",
    "hummingbird",
    "jacamar",
    "toucan",
    "duck",
    "red-breasted merganser",
    "goose",
    "black swan",
    "tusker",
    "echidna",
    "platypus",
    "wallaby",
    "koala",
    "wombat",
    "jellyfish",
    "sea anemone",
    "brain coral",
    "flatworm",
    "nematode",
    "conch",
    "snail",
    "slug",
    "sea slug",
    "chiton",
    "chambered nautilus",
    "Dungeness crab",
    "rock crab",
    "fiddler crab",
    "red king crab",
    "American lobster",
    "spiny lobster",
    "crayfish",
    "hermit crab",
    "isopod",
    "white stork",
    "black stork",
    "spoonbill",
    "flamingo",
    "little blue heron",
    "great egret",
    "bittern bird",
    "crane bird",
    "limpkin",
    "common gallinule",
    "American coot",
    "bustard",
    "ruddy turnstone",
    "dunlin",
    "common redshank",
    "dowitcher",
    "oystercatcher",
    "pelican",
    "king penguin",
    "albatross",
    "grey whale",
    "killer whale",
    "dugong",
    "sea lion",
    "Chihuahua",
    "Japanese Chin",
    "Maltese",
    "Pekingese",
    "Shih Tzu",
    "King Charles Spaniel",
    "Papillon",
    "toy terrier",
    "Rhodesian Ridgeback",
    "Afghan Hound",
    "Basset Hound",
    "Beagle",
    "Bloodhound",
    "Bluetick Coonhound",
    "Black and Tan Coonhound",
    "Treeing Walker Coonhound",
    "English foxhound",
    "Redbone Coonhound",
    "borzoi",
    "Irish Wolfhound",
    "Italian Greyhound",
    "Whippet",
    "Ibizan Hound",
    "Norwegian Elkhound",
    "Otterhound",
    "Saluki",
    "Scottish Deerhound",
    "Weimaraner",
    "Staffordshire Bull Terrier",
    "American Staffordshire Terrier",
    "Bedlington Terrier",
    "Border Terrier",
    "Kerry Blue Terrier",
    "Irish Terrier",
    "Norfolk Terrier",
    "Norwich Terrier",
    "Yorkshire Terrier",
    "Wire Fox Terrier",
    "Lakeland Terrier",
    "Sealyham Terrier",
    "Airedale Terrier",
    "Cairn Terrier",
    "Australian Terrier",
    "Dandie Dinmont Terrier",
    "Boston Terrier",
    "Miniature Schnauzer",
    "Giant Schnauzer",
    "Standard Schnauzer",
    "Scottish Terrier",
    "Tibetan Terrier",
    "Australian Silky Terrier",
    "Soft-coated Wheaten Terrier",
    "West Highland White Terrier",
    "Lhasa Apso",
    "Flat-Coated Retriever",
    "Curly-coated Retriever",
    "Golden Retriever",
    "Labrador Retriever",
    "Chesapeake Bay Retriever",
    "German Shorthaired Pointer",
    "Vizsla",
    "English Setter",
    "Irish Setter",
    "Gordon Setter",
    "Brittany dog",
    "Clumber Spaniel",
    "English Springer Spaniel",
    "Welsh Springer Spaniel",
    "Cocker Spaniel",
    "Sussex Spaniel",
    "Irish Water Spaniel",
    "Kuvasz",
    "Schipperke",
    "Groenendael dog",
    "Malinois",
    "Briard",
    "Australian Kelpie",
    "Komondor",
    "Old English Sheepdog",
    "Shetland Sheepdog",
    "collie",
    "Border Collie",
    "Bouvier des Flandres dog",
    "Rottweiler",
    "German Shepherd Dog",
    "Dobermann",
    "Miniature Pinscher",
    "Greater Swiss Mountain Dog",
    "Bernese Mountain Dog",
    "Appenzeller Sennenhund",
    "Entlebucher Sennenhund",
    "Boxer",
    "Bullmastiff",
    "Tibetan Mastiff",
    "French Bulldog",
    "Great Dane",
    "St. Bernard",
    "husky",
    "Alaskan Malamute",
    "Siberian Husky",
    "Dalmatian",
    "Affenpinscher",
    "Basenji",
    "pug",
    "Leonberger",
    "Newfoundland dog",
    "Great Pyrenees dog",
    "Samoyed",
    "Pomeranian",
    "Chow Chow",
    "Keeshond",
    "brussels griffon",
    "Pembroke Welsh Corgi",
    "Cardigan Welsh Corgi",
    "Toy Poodle",
    "Miniature Poodle",
    "Standard Poodle",
    "Mexican hairless dog (xoloitzcuintli)",
    "grey wolf",
    "Alaskan tundra wolf",
    "red wolf or maned wolf",
    "coyote",
    "dingo",
    "dhole",
    "African wild dog",
    "hyena",
    "red fox",
    "kit fox",
    "Arctic fox",
    "grey fox",
    "tabby cat",
    "tiger cat",
    "Persian cat",
    "Siamese cat",
    "Egyptian Mau",
    "cougar",
    "lynx",
    "leopard",
    "snow leopard",
    "jaguar",
    "lion",
    "tiger",
    "cheetah",
    "brown bear",
    "American black bear",
    "polar bear",
    "sloth bear",
    "mongoose",
    "meerkat",
    "tiger beetle",
    "ladybug",
    "ground beetle",
    "longhorn beetle",
    "leaf beetle",
    "dung beetle",
    "rhinoceros beetle",
    "weevil",
    "fly",
    "bee",
    "ant",
    "grasshopper",
    "cricket insect",
    "stick insect",
    "cockroach",
    "praying mantis",
    "cicada",
    "leafhopper",
    "lacewing",
    "dragonfly",
    "damselfly",
    "red admiral butterfly",
    "ringlet butterfly",
    "monarch butterfly",
    "small white butterfly",
    "sulphur butterfly",
    "gossamer-winged butterfly",
    "starfish",
    "sea urchin",
    "sea cucumber",
    "cottontail rabbit",
    "hare",
    "Angora rabbit",
    "hamster",
    "porcupine",
    "fox squirrel",
    "marmot",
    "beaver",
    "guinea pig",
    "common sorrel horse",
    "zebra",
    "pig",
    "wild boar",
    "warthog",
    "hippopotamus",
    "ox",
    "water buffalo",
    "bison",
    "ram (adult male sheep)",
    "bighorn sheep",
    "Alpine ibex",
    "hartebeest",
    "impala (antelope)",
    "gazelle",
    "arabian camel",
    "llama",
    "weasel",
    "mink",
    "European polecat",
    "black-footed ferret",
    "otter",
    "skunk",
    "badger",
    "armadillo",
    "three-toed sloth",
    "orangutan",
    "gorilla",
    "chimpanzee",
    "gibbon",
    "siamang",
    "guenon",
    "patas monkey",
    "baboon",
    "macaque",
    "langur",
    "black-and-white colobus",
    "proboscis monkey",
    "marmoset",
    "white-headed capuchin",
    "howler monkey",
    "titi monkey",
    "Geoffroy's spider monkey",
    "common squirrel monkey",
    "ring-tailed lemur",
    "indri",
    "Asian elephant",
    "African bush elephant",
    "red panda",
    "giant panda",
    "snoek fish",
    "eel",
    "silver salmon",
    "rock beauty fish",
    "clownfish",
    "sturgeon",
    "gar fish",
    "lionfish",
    "pufferfish",
    "abacus",
    "abaya",
    "academic gown",
    "accordion",
    "acoustic guitar",
    "aircraft carrier",
    "airliner",
    "airship",
    "altar",
    "ambulance",
    "amphibious vehicle",
    "analog clock",
    "apiary",
    "apron",
    "trash can",
    "assault rifle",
    "backpack",
    "bakery",
    "balance beam",
    "balloon",
    "ballpoint pen",
    "Band-Aid",
    "banjo",
    "baluster / handrail",
    "barbell",
    "barber chair",
    "barbershop",
    "barn",
    "barometer",
    "barrel",
    "wheelbarrow",
    "baseball",
    "basketball",
    "bassinet",
    "bassoon",
    "swimming cap",
    "bath towel",
    "bathtub",
    "station wagon",
    "lighthouse",
    "beaker",
    "military hat (bearskin or shako)",
    "beer bottle",
    "beer glass",
    "bell tower",
    "baby bib",
    "tandem bicycle",
    "bikini",
    "ring binder",
    "binoculars",
    "birdhouse",
    "boathouse",
    "bobsleigh",
    "bolo tie",
    "poke bonnet",
    "bookcase",
    "bookstore",
    "bottle cap",
    "hunting bow",
    "bow tie",
    "brass memorial plaque",
    "bra",
    "breakwater",
    "breastplate",
    "broom",
    "bucket",
    "buckle",
    "bulletproof vest",
    "high-speed train",
    "butcher shop",
    "taxicab",
    "cauldron",
    "candle",
    "cannon",
    "canoe",
    "can opener",
    "cardigan",
    "car mirror",
    "carousel",
    "tool kit",
    "cardboard box / carton",
    "car wheel",
    "automated teller machine",
    "cassette",
    "cassette player",
    "castle",
    "catamaran",
    "CD player",
    "cello",
    "mobile phone",
    "chain",
    "chain-link fence",
    "chain mail",
    "chainsaw",
    "storage chest",
    "chiffonier",
    "bell or wind chime",
    "china cabinet",
    "Christmas stocking",
    "church",
    "movie theater",
    "cleaver",
    "cliff dwelling",
    "cloak",
    "clogs",
    "cocktail shaker",
    "coffee mug",
    "coffeemaker",
    "spiral or coil",
    "combination lock",
    "computer keyboard",
    "candy store",
    "container ship",
    "convertible",
    "corkscrew",
    "cornet",
    "cowboy boot",
    "cowboy hat",
    "cradle",
    "construction crane",
    "crash helmet",
    "crate",
    "infant bed",
    "Crock Pot",
    "croquet ball",
    "crutch",
    "cuirass",
    "dam",
    "desk",
    "desktop computer",
    "rotary dial telephone",
    "diaper",
    "digital clock",
    "digital watch",
    "dining table",
    "dishcloth",
    "dishwasher",
    "disc brake",
    "dock",
    "dog sled",
    "dome",
    "doormat",
    "drilling rig",
    "drum",
    "drumstick",
    "dumbbell",
    "Dutch oven",
    "electric fan",
    "electric guitar",
    "electric locomotive",
    "entertainment center",
    "envelope",
    "espresso machine",
    "face powder",
    "feather boa",
    "filing cabinet",
    "fireboat",
    "fire truck",
    "fire screen",
    "flagpole",
    "flute",
    "folding chair",
    "football helmet",
    "forklift",
    "fountain",
    "fountain pen",
    "four-poster bed",
    "freight car",
    "French horn",
    "frying pan",
    "fur coat",
    "garbage truck",
    "gas mask or respirator",
    "gas pump",
    "goblet",
    "go-kart",
    "golf ball",
    "golf cart",
    "gondola",
    "gong",
    "gown",
    "grand piano",
    "greenhouse",
    "radiator grille",
    "grocery store",
    "guillotine",
    "hair clip",
    "hair spray",
    "half-track",
    "hammer",
    "hamper",
    "hair dryer",
    "hand-held computer",
    "handkerchief",
    "hard disk drive",
    "harmonica",
    "harp",
    "combine harvester",
    "hatchet",
    "holster",
    "home theater",
    "honeycomb",
    "hook",
    "hoop skirt",
    "gymnastic horizontal bar",
    "horse-drawn vehicle",
    "hourglass",
    "iPod",
    "clothes iron",
    "carved pumpkin",
    "jeans",
    "jeep",
    "T-shirt",
    "jigsaw puzzle",
    "rickshaw",
    "joystick",
    "kimono",
    "knee pad",
    "knot",
    "lab coat",
    "ladle",
    "lampshade",
    "laptop computer",
    "lawn mower",
    "lens cap",
    "letter opener",
    "library",
    "lifeboat",
    "lighter",
    "limousine",
    "ocean liner",
    "lipstick",
    "slip-on shoe",
    "lotion",
    "music speaker",
    "loupe magnifying glass",
    "sawmill",
    "magnetic compass",
    "messenger bag",
    "mailbox",
    "tights",
    "one-piece bathing suit",
    "manhole cover",
    "maraca",
    "marimba",
    "mask",
    "matchstick",
    "maypole",
    "maze",
    "measuring cup",
    "medicine cabinet",
    "megalith",
    "microphone",
    "microwave oven",
    "military uniform",
    "milk can",
    "minibus",
    "miniskirt",
    "minivan",
    "missile",
    "mitten",
    "mixing bowl",
    "mobile home",
    "ford model t",
    "modem",
    "monastery",
    "monitor",
    "moped",
    "mortar and pestle",
    "graduation cap",
    "mosque",
    "mosquito net",
    "vespa",
    "mountain bike",
    "tent",
    "computer mouse",
    "mousetrap",
    "moving van",
    "muzzle",
    "metal nail",
    "neck brace",
    "necklace",
    "baby pacifier",
    "notebook computer",
    "obelisk",
    "oboe",
    "ocarina",
    "odometer",
    "oil filter",
    "pipe organ",
    "oscilloscope",
    "overskirt",
    "bullock cart",
    "oxygen mask",
    "product packet / packaging",
    "paddle",
    "paddle wheel",
    "padlock",
    "paintbrush",
    "pajamas",
    "palace",
    "pan flute",
    "paper towel",
    "parachute",
    "parallel bars",
    "park bench",
    "parking meter",
    "railroad car",
    "patio",
    "payphone",
    "pedestal",
    "pencil case",
    "pencil sharpener",
    "perfume",
    "Petri dish",
    "photocopier",
    "plectrum",
    "Pickelhaube",
    "picket fence",
    "pickup truck",
    "pier",
    "piggy bank",
    "pill bottle",
    "pillow",
    "ping-pong ball",
    "pinwheel",
    "pirate ship",
    "drink pitcher",
    "block plane",
    "planetarium",
    "plastic bag",
    "plate rack",
    "farm plow",
    "plunger",
    "Polaroid camera",
    "pole",
    "police van",
    "poncho",
    "pool table",
    "soda bottle",
    "plant pot",
    "potter's wheel",
    "power drill",
    "prayer rug",
    "printer",
    "prison",
    "missile",
    "projector",
    "hockey puck",
    "punching bag",
    "purse",
    "quill",
    "quilt",
    "race car",
    "racket",
    "radiator",
    "radio",
    "radio telescope",
    "rain barrel",
    "recreational vehicle",
    "fishing casting reel",
    "reflex camera",
    "refrigerator",
    "remote control",
    "restaurant",
    "revolver",
    "rifle",
    "rocking chair",
    "rotisserie",
    "eraser",
    "rugby ball",
    "ruler measuring stick",
    "sneaker",
    "safe",
    "safety pin",
    "salt shaker",
    "sandal",
    "sarong",
    "saxophone",
    "scabbard",
    "weighing scale",
    "school bus",
    "schooner",
    "scoreboard",
    "CRT monitor",
    "screw",
    "screwdriver",
    "seat belt",
    "sewing machine",
    "shield",
    "shoe store",
    "shoji screen / room divider",
    "shopping basket",
    "shopping cart",
    "shovel",
    "shower cap",
    "shower curtain",
    "ski",
    "balaclava ski mask",
    "sleeping bag",
    "slide rule",
    "sliding door",
    "slot machine",
    "snorkel",
    "snowmobile",
    "snowplow",
    "soap dispenser",
    "soccer ball",
    "sock",
    "solar thermal collector",
    "sombrero",
    "soup bowl",
    "keyboard space bar",
    "space heater",
    "space shuttle",
    "spatula",
    "motorboat",
    "spider web",
    "spindle",
    "sports car",
    "spotlight",
    "stage",
    "steam locomotive",
    "through arch bridge",
    "steel drum",
    "stethoscope",
    "scarf",
    "stone wall",
    "stopwatch",
    "stove",
    "strainer",
    "tram",
    "stretcher",
    "couch",
    "stupa",
    "submarine",
    "suit",
    "sundial",
    "sunglasses",
    "sunglasses",
    "sunscreen",
    "suspension bridge",
    "mop",
    "sweatshirt",
    "swim trunks / shorts",
    "swing",
    "electrical switch",
    "syringe",
    "table lamp",
    "tank",
    "tape player",
    "teapot",
    "teddy bear",
    "television",
    "tennis ball",
    "thatched roof",
    "front curtain",
    "thimble",
    "threshing machine",
    "throne",
    "tile roof",
    "toaster",
    "tobacco shop",
    "toilet seat",
    "torch",
    "totem pole",
    "tow truck",
    "toy store",
    "tractor",
    "semi-trailer truck",
    "tray",
    "trench coat",
    "tricycle",
    "trimaran",
    "tripod",
    "triumphal arch",
    "trolleybus",
    "trombone",
    "hot tub",
    "turnstile",
    "typewriter keyboard",
    "umbrella",
    "unicycle",
    "upright piano",
    "vacuum cleaner",
    "vase",
    "vaulted or arched ceiling",
    "velvet fabric",
    "vending machine",
    "vestment",
    "viaduct",
    "violin",
    "volleyball",
    "waffle iron",
    "wall clock",
    "wallet",
    "wardrobe",
    "military aircraft",
    "sink",
    "washing machine",
    "water bottle",
    "water jug",
    "water tower",
    "whiskey jug",
    "whistle",
    "hair wig",
    "window screen",
    "window shade",
    "Windsor tie",
    "wine bottle",
    "airplane wing",
    "wok",
    "wooden spoon",
    "wool",
    "split-rail fence",
    "shipwreck",
    "sailboat",
    "yurt",
    "website",
    "comic book",
    "crossword",
    "traffic or street sign",
    "traffic light",
    "dust jacket",
    "menu",
    "plate",
    "guacamole",
    "consomme",
    "hot pot",
    "trifle",
    "ice cream",
    "popsicle",
    "baguette",
    "bagel",
    "pretzel",
    "cheeseburger",
    "hot dog",
    "mashed potatoes",
    "cabbage",
    "broccoli",
    "cauliflower",
    "zucchini",
    "spaghetti squash",
    "acorn squash",
    "butternut squash",
    "cucumber",
    "artichoke",
    "bell pepper",
    "cardoon",
    "mushroom",
    "Granny Smith apple",
    "strawberry",
    "orange",
    "lemon",
    "fig",
    "pineapple",
    "banana",
    "jackfruit",
    "cherimoya (custard apple)",
    "pomegranate",
    "hay",
    "carbonara",
    "chocolate syrup",
    "dough",
    "meatloaf",
    "pizza",
    "pot pie",
    "burrito",
    "red wine",
    "espresso",
    "tea cup",
    "eggnog",
    "mountain",
    "bubble",
    "cliff",
    "coral reef",
    "geyser",
    "lakeshore",
    "promontory",
    "sandbar",
    "beach",
    "valley",
    "volcano",
    "baseball player",
    "bridegroom",
    "scuba diver",
    "rapeseed",
    "daisy",
    "yellow lady's slipper",
    "corn",
    "acorn",
    "rose hip",
    "horse chestnut seed",
    "coral fungus",
    "agaric",
    "gyromitra",
    "stinkhorn mushroom",
    "earth star fungus",
    "hen of the woods mushroom",
    "bolete",
    "corn cob",
    "toilet paper",
]

IMAGENET_PROMPTS = [
    "itap of a {}.",
    "a photo of the small {}.",
    "a photo of the large {}.",
    "a bad photo of the {}.",
    "a origami {}.",
    "a {} in a video game.",
    "art of the {}.",
]


class CLIPPreprocessor(PreProcessor):
    """A pre-processor for image data using the CLIP model."""

    def __init__(self, image_dimensions: (int, int), image_only: bool = True, image_input_index: int = 0):
        """Initialize the CLIPPreprocessor.

        Args:
            image_dimensions (tuple[int, int]): The desired dimensions of the output images.
            image_only (bool): Whether to process only image tokens. Defaults to True.
            image_input_index (int): The index of the image input in the input_data representation.
                                     Defaults to 0.
        """
        transformers = Helper.safe_import_package("transformers", "4.44.0")
        if transformers:
            self.processor = transformers.CLIPProcessor.from_pretrained("openai/clip-vit-base-patch16")
        else:
            raise ImportError(
                "Failed to load CLIPProcessor using transformers library. "
                "Please install the required version of transformers."
            )
        self.image_dimensions = image_dimensions
        self.image_only = image_only
        self.image_input_index = image_input_index
        self.validate()

    def validate(self):
        """Validate whether the parameters have been set correctly.

        Raises:
            ValueError: If image_only is not a boolean value.
            ValueError: If image dimensions are not integers or negative values.
        """
        if not isinstance(self.image_only, bool):
            raise ValueError(f"image_only must be a boolean value. Got {self.image_only} instead.")
        if not len(self.image_dimensions) == 2:
            raise ValueError(
                "Invalid image dimensions provided. Image dimensions must be provided as "
                "(image_height, image_width)"
            )
        if not all((isinstance(dim, int) and dim > 0) for dim in self.image_dimensions):
            raise ValueError(
                "Invalid image dimensions provided. Both width and height should be"
                " postive integers greater than 0."
            )

    def execute(self, input_data: ImageRepresentation) -> ImageRepresentation:
        """Execute the pre-processing of the input data.

        Args:
            input_data (ImageRepresentation): The input data to process.

        Returns:
            ImageRepresentation: The processed data.
        """
        Helper.safe_import_package("PIL", "10.2.0")
        from PIL import Image

        # Limit number of classes and prompts to valid values
        num_base_class = len(IMAGENET_CLASSES)
        num_prompt = len(IMAGENET_PROMPTS)

        # Check if image_input_index is within range of input data
        if len(input_data.data) < self.image_input_index:
            raise RuntimeError(
                f"image_input_index {self.image_input_index} is out of range."
                f" It should be less than the number of data items provided in input_data "
                f"representation {len(input_data.data)}."
            )

        # Determine whether to process image only or with text prompts
        if self.image_only:
            num_prompt = 1
            num_base_class = 1
            imagenet_prompts_l = ["{}"]  # single prompt for dummy class
            imagenet_classes_l = ["dummy"]  # single class
        else:
            imagenet_prompts_l = IMAGENET_PROMPTS[0:num_prompt]  # select valid prompts
            imagenet_classes_l = IMAGENET_CLASSES[0:num_base_class]  # select valid classes

        # Create text prompts by formatting class names into prompt strings
        text = [x.format(y) for y in imagenet_classes_l for x in imagenet_prompts_l]

        # Calculate total number of classes and batch size
        num_class = num_base_class * num_prompt
        batch_size = 1

        # Get image dimensions
        h, w = self.image_dimensions
        image = Image.fromarray(np.uint8(input_data.data[self.image_input_index])).convert("RGB")
        imw, imh = image.size
        if imw > imh:
            width = w
            height = int(math.ceil((w * imh) / imw))
        else:
            height = h
            width = int(math.ceil((h * imw) / imw))
        size = (width, height)
        image = image.resize(size, Image.BILINEAR)

        # Process input data using the processor
        inputs = self.processor(text=text, images=(image,), return_tensors="pt", padding=True)

        # Determine which keys to extract from processed inputs
        if self.image_only:
            key_list = ["pixel_values"]
        else:
            key_list = [
                "pixel_values",
                "input_ids",
                "attention_mask",
                "causal_attention_mask",
                "class_embeds",
            ]
            seq_len = inputs["input_ids"].shape[1]
            causal_attention_mask = self.generate_causal_attention_mask(num_class, seq_len)
            class_embeds = self.generate_class_embeds(batch_size)
            inputs["causal_attention_mask"] = causal_attention_mask
            inputs["class_embeds"] = class_embeds

        # Extract specified keys from processed inputs and store in input data representation
        out_data = []
        for key in key_list:
            out_data.append(inputs[key].detach().numpy())
        input_data.data = out_data
        input_data.metadata["input_keys"] = key_list
        return input_data

    @staticmethod
    def generate_causal_attention_mask(num_channels: int, seq_len: int) -> "torch.Tensor":  # noqa: F821
        """Generate a causal attention mask for a given sequence length and number of heads.

        Args:
            num_channels (int): The number of channels.
            seq_len (int): The sequence length.

        Returns:
            torch.Tensor: A tensor representing the causal attention mask.
        """
        torch = Helper.safe_import_package("torch", "2.4.0")
        mask = torch.empty(num_channels, seq_len, seq_len)
        mask.fill_(float("-inf"))
        mask.triu_(1)  # zero out the lower diagonal
        mask = mask.unsqueeze(1)  # expand mask
        return mask

    @staticmethod
    def generate_class_embeds(batch_size: int) -> "torch.Tensor":  # noqa: F821
        """Generate a tensor of random class embeddings.

        Args:
            batch_size (int): Batch size.

        Returns:
            torch.Tensor: A tensor with random values drawn from a normal distribution.
                Shape: [batch_size, 1, 768]
        """
        # Import the PyTorch library and set a fixed seed for reproducibility
        torch = Helper.safe_import_package("torch", "2.4.0")
        torch.random.manual_seed(123)

        # Generate a tensor of random class embeddings with shape [bs, 1, 768]
        # Using a normal distribution with mean 0 and standard deviation 1
        class_embeds = torch.randn(batch_size, 1, 768)
        return class_embeds
