# ==============================================================================
#
#  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
#  All rights reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# ==============================================================================
"""
This module provides the pass for reordering 
(FastHadamardTransform->GroupSlice) -> (GroupSlice->FastHadamardTransform)
"""



from onnxscript import ir

from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.base_rewriter import M2sBaseRewriter
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.utils import (
    get_gslice_attrs,
    is_reorderable_group_slice,
)
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .utils.utils import (
    get_value_numeric_shape,
    has_static_shape_on_value,
    logger,
)


class M2sReorderFastHadamardTransformGroupslice(M2sBaseRewriter):
    '''
    Reorder subgraph:
        Subgraph(in_a) --> b,b0,b1,b2
        {
            b = FastHadamardTransform(in_a) # such as FastHadamardTransform
            b0,b1,b2... = GroupSlice(b)
        }
    Into:
        Subgraph(in_a) --> b,b0,b1,b2
        {
            a0,a1,a2... = GroupSlice(in_a)

            b0 = FastHadamardTransform(a0)
            b1 = FastHadamardTransform(a1)
            b2 = FastHadamardTransform(a2)
            ...

            # if possible
            b = concat(b0,b1,b2,...) 
            # or b = FastHadamardTransform(in_a)
        }

    Also, encodings are updated
    '''

    def __init__(self, graph: ir.Graph):
        super().__init__(graph)
        self.support_op_types = ["FastHadamardTransform"]
        self.op_node: ir.Node | None = None

    def match(self, node: ir.Node) -> bool:
        gslice_node = node
        if not is_reorderable_group_slice(gslice_node):
            return False
        # check for mypy, definitely true
        assert gslice_node.inputs[0] is not None
        op_node = gslice_node.inputs[0].producer()
        assert op_node is not None  # check for mypy, definitely true
        if op_node.op_type not in self.support_op_types:
            return False
        if op_node.inputs[0] is None:
            return False
        
        if not has_static_shape_on_value(op_node.inputs[0]):
            return False
        
        groupslice_attrs = get_gslice_attrs(gslice_node)
        
        # we can't reorder if slice_axis is the last axis
        output_shape = get_value_numeric_shape(op_node.outputs[0])
        if groupslice_attrs.axis == len(output_shape) - 1:
            return False

        self.op_node = op_node
        return True

    def rewrite(self, node: ir.Node) -> bool:
        gslice_node = node
        assert self.op_node is not None  # check for mypy, definitely true

        a_groupslice_attrs = get_gslice_attrs(gslice_node)

        _ = self.rewrite_based_on_gslice_attrs(
            self.op_node, [gslice_node],
            inputs_gslice_attrs=[a_groupslice_attrs]
        )

        logger.debug("applied pass %s on '%s'",
                      self.get_curr_pass_name(), self.op_node.name)
        self.op_node = None
        return True
