# ==============================================================================
#
#  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
#  All rights reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# ==============================================================================
"""
This module provides the pass for reordering 
(LayerNormalization->GroupSlice) -> (GroupSlice->LayerNormalization)
"""



from onnxscript import ir

from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.base_rewriter import M2sBaseRewriter
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.utils import (
    FullGroupSliceAttrs,
    get_gslice_attrs,
    is_reorderable_group_slice,
)
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .utils.utils import (
    convert_attr_to_py,
    get_value_numeric_shape,
    has_static_shape_on_value,
    logger,
)


class M2sReorderLayernormGroupslice(M2sBaseRewriter):
    '''
    Reorder subgraph:
        Subgraph(in_a) --> b,b0,b1,b2
        {
            b = Layernorm(in_a) # such as Layernorm
            b0,b1,b2... = GroupSlice(b)
        }
    Into:
        Subgraph(in_a) --> b,b0,b1,b2
        {
            a0,a1,a2... = GroupSlice(in_a)

            b0 = Layernorm(a0)
            b1 = Layernorm(a1)
            b2 = Layernorm(a2)
            ...

            # if possible
            b = concat(b0,b1,b2,...) 
            # or b = Layernorm(in_a)
        }

    Also, encodings are updated
    '''

    def __init__(self, graph: ir.Graph):
        super().__init__(graph)
        self.support_op_types = ["LayerNormalization"]
        self.op_node: ir.Node | None = None

    def match(self, node: ir.Node) -> bool:
        gslice_node = node
        if not is_reorderable_group_slice(gslice_node):
            return False
        # check for mypy, definitely true
        assert gslice_node.inputs[0] is not None
        op_node = gslice_node.inputs[0].producer()
        assert op_node is not None  # check for mypy, definitely true
        if op_node.op_type not in self.support_op_types:
            return False

        assert op_node.inputs[0] is not None # check for mypy, definitely true
        if not has_static_shape_on_value(op_node.inputs[0]):
            return False

        reduce_axis = convert_attr_to_py(op_node.attributes["axis"],
                                               "as_int")
        if reduce_axis < 0:
            reduce_axis += len(get_value_numeric_shape(op_node.inputs[0]))

        if reduce_axis == convert_attr_to_py(gslice_node.attributes["axis"],
                                                   "as_int"):
            return False

        self.op_node = op_node
        return True

    def rewrite(self, node: ir.Node) -> bool:
        gslice_node = node
        assert self.op_node is not None  # check for mypy, definitely true

        # when reduce_axis != gslice_axis
        # input gslice attrs should be the same on output gslice
        a_groupslice_attrs = get_gslice_attrs(gslice_node)

        other_inputs_gslice_attrs = []
        for _ in self.op_node.inputs[1:]:
            # scale and bias
            other_inputs_gslice_attrs.append(
                FullGroupSliceAttrs(a_groupslice_attrs.num_outputs(),
                                    a_groupslice_attrs.head_slice_ids[:],
                                    a_groupslice_attrs.batch_slice_ids[:])
            )
        _ = self.rewrite_based_on_gslice_attrs(
            self.op_node, [gslice_node],
            inputs_gslice_attrs=[a_groupslice_attrs] +
            other_inputs_gslice_attrs
        )

        logger.debug("applied pass %s on '%s'",
                      self.get_curr_pass_name(), self.op_node.name)
        self.op_node = None
        return True
