# ==============================================================================
#
#  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
#  All rights reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# ==============================================================================
"""
This module provides the pass for reordering 
(Transpose->GroupSlice) -> (GroupSlice->Transpose)
"""


from onnxscript import ir

from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.base_rewriter import M2sBaseRewriter
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.utils import (
    get_gslice_attrs,
    is_reorderable_group_slice,
)
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .utils.utils import (
    check_static_shape_of_node_io,
    convert_attr_to_py,
    logger,
)


class M2sReorderTransposeGroupslice(M2sBaseRewriter):
    '''
    Transform subgraph:
        Subgraph(in_a) --> b,b0,b1,b2
        {
            b = Transpose(in_a)
            b0,b1,b2... = GroupSlice(b)
        }
    Into:
        Subgraph(in_a) --> b,b0,b1,b2
        {

            a0,a1,a2... = GroupSlice(in_a)

            b0 = Transpose(a0)
            b1 = Transpose(a1)
            b2 = Transpose(a2)
            ...

            # if possible
            b = concat(b0,b1,b2,...) 
            # or b = Transpose(in_a)
        }

    Also, encodings are updated
    '''

    def __init__(self, graph: ir.Graph):
        super().__init__(graph)
        self.op_node: ir.Node | None = None

    def match(self, node: ir.Node) -> bool:
        gslice_node = node
        if not is_reorderable_group_slice(gslice_node):
            return False
        # check for mypy, definitely true
        assert gslice_node.inputs[0] is not None
        op_node = gslice_node.inputs[0].producer()
        assert op_node is not None  # check for mypy, definitely true
        if op_node.op_type != "Transpose":
            return False
        check_static_shape_of_node_io(op_node)

        self.op_node = op_node
        return True

    def rewrite(self, node: ir.Node) -> bool:
        gslice_node = node
        assert self.op_node is not None  # check for mypy, definitely true
        perm = convert_attr_to_py(self.op_node.attributes["perm"],
                                        "as_ints")

        a_groupslice_attrs = get_gslice_attrs(gslice_node)
        a_groupslice_attrs.axis = perm[a_groupslice_attrs.axis]

        _ = self.rewrite_based_on_gslice_attrs(
            self.op_node, [gslice_node],
            inputs_gslice_attrs=[a_groupslice_attrs]
        )

        logger.debug("applied pass %s on '%s'",
                      self.get_curr_pass_name(), self.op_node.name)

        self.op_node = None
        return True
