# ==============================================================================
#
#  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
#  All rights reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# ==============================================================================
"""
This module provides the pass for reordering 
(UnaryOp->GroupSlice) -> (GroupSlice->UnaryOp)
"""


from onnxscript import ir

from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.base_rewriter import M2sBaseRewriter
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.utils import (
    get_gslice_attrs,
    is_reorderable_group_slice,
)
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .utils.utils import (
    check_static_shape_of_node_io,
    logger,
)


class M2sReorderUnaryGroupslice(M2sBaseRewriter):
    '''
    Transform subgraph:
        Subgraph(in_a) --> c, c0,c1,c2...
        {
            c = UnaryOp(in_a)
            c0,c1,c2... = GroupSlice(c)
        }
    Into:
        Subgraph(in_a, in_b) --> c, c0,c1,c2...
        {
            a0,a1,a2,... = GroupSlice(in_a)

            c0 = UnaryOp(in_a0)
            c1 = UnaryOp(in_a1)
            c2 = UnaryOp(in_a2)

            # if possible
            c = concat(c0,c1,c2,...) 
            # or c = UnaryOp(in_a)

        }

    Broadcasting will be automatically updated
    Also, encodings are updated
    '''

    def __init__(self, graph: ir.Graph):
        super().__init__(graph)
        self.support_op_types = ["Neg", "Sqrt", "Cast", "Erf", "Sigmoid", "Relu", "Gelu", "Tanh"]
        self.op_node: ir.Node | None = None

    def match(self, node: ir.Node) -> bool:
        gslice_node = node
        if not is_reorderable_group_slice(gslice_node):
            return False
        # check for mypy, definitely true
        assert gslice_node.inputs[0] is not None
        op_node = gslice_node.inputs[0].producer()
        assert op_node is not None  # check for mypy, definitely true
        if op_node.op_type not in self.support_op_types:
            return False

        check_static_shape_of_node_io(op_node)
        self.op_node = op_node
        return True

    def rewrite(self, node: ir.Node) -> bool:
        assert self.op_node is not None  # check for mypy, definitely true
        gslice_node = node
        in_a_gslice_attrs = get_gslice_attrs(gslice_node)

        _ = self.rewrite_based_on_gslice_attrs(
            self.op_node, [gslice_node],
            inputs_gslice_attrs=[in_a_gslice_attrs]
        )

        logger.debug("applied pass %s on '%s'",
                      self.get_curr_pass_name(), self.op_node.name)

        self.op_node = None
        return True
