# ==============================================================================
#
#  Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
#  All rights reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# ==============================================================================
"""
This module provides the pass for reordering 
(UnaryOp->GroupSlice) -> (GroupSlice->UnaryOp)
"""


from onnxscript import ir

from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.base_rewriter import M2sBaseRewriter
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .passes.mha2sha.utils import (
    BroadcastHelper,
    get_gslice_attrs,
    is_reorderable_group_slice,
)
from qti.aisw.tools.core.utilities.framework.frameworks.onnx.transform.v2.mha2sha\
    .utils.utils import (
    check_static_shape_of_node_io,
    get_value_numeric_shape,
    logger,
)


class M2sReorderWhereGroupslice(M2sBaseRewriter):
    '''
    Transform subgraph:
        Subgraph(in_cond, in_a, in_b) --> c, c0,c1,c2...
        {
            c = Where(in_cond, in_a, in_b)
            c0,c1,c2... = GroupSlice(c)
        }
    Into:
        Subgraph(in_cond, in_a, in_b) --> c, c0,c1,c2...
        {
            con1,con2,con3,... = GroupSlice(in_cond)
            a0,a1,a2,... = GroupSlice(in_a)
            b0,b1,b2,... = GroupSlice(in_b)

            c0 = Where(con1, in_a0, in_b0)
            c1 = Where(con2, in_a1, in_b1)
            c2 = Where(con3, in_a2, in_b2)

            # if possible
            c = concat(c0,c1,c2,...) 
            # or c = Where(in_a, in_b)

        }

    Broadcasting will be automatically updated
    Also, encodings are updated
    '''

    def __init__(self, graph: ir.Graph):
        super().__init__(graph)
        self.op_node: ir.Node | None = None

    def match(self, node: ir.Node) -> bool:
        gslice_node = node
        if not is_reorderable_group_slice(gslice_node):
            return False
        # check for mypy, definitely true
        assert gslice_node.inputs[0] is not None
        op_node = gslice_node.inputs[0].producer()
        assert op_node is not None  # check for mypy, definitely true
        if op_node.op_type != "Where":
            return False

        check_static_shape_of_node_io(op_node)
        self.op_node = op_node
        return True

    def rewrite(self, node: ir.Node) -> bool:
        gslice_node = node
        assert self.op_node is not None  # check for mypy, definitely true
        output_shape = get_value_numeric_shape(self.op_node.outputs[0])

        # the broadcast is multidirectional among condition/a/b
        in_cond_bc_helper = BroadcastHelper(
            get_value_numeric_shape(
                self.op_node.inputs[0]), output_shape, output_shape
        )
        in_a_bc_helper = BroadcastHelper(
            get_value_numeric_shape(
                self.op_node.inputs[1]), output_shape, output_shape
        )
        in_b_bc_helper = BroadcastHelper(
            get_value_numeric_shape(
                self.op_node.inputs[2]), output_shape, output_shape
        )

        gslice_attrs = get_gslice_attrs(gslice_node)

        cond_groupslice_attrs = in_cond_bc_helper.get_input_group_attrs(
            0, gslice_attrs)
        a_groupslice_attrs = in_a_bc_helper.get_input_group_attrs(
            0, gslice_attrs)
        b_groupslice_attrs = in_b_bc_helper.get_input_group_attrs(
            0, gslice_attrs)

        _ = self.rewrite_based_on_gslice_attrs(
            self.op_node, [gslice_node],
            inputs_gslice_attrs=[cond_groupslice_attrs,
                                 a_groupslice_attrs, b_groupslice_attrs]
        )

        logger.debug("applied pass %s on '%s'",
                      self.get_curr_pass_name(), self.op_node.name)
        self.op_node = None
        return True
