# =============================================================================
#
#  Copyright (c) 2021 Qualcomm Technologies, Inc.
#  All Rights Reserved.
#  Confidential and Proprietary - Qualcomm Technologies, Inc.
#
# =============================================================================

import sys
import json
import pandas as pd
import numpy as np
from ParserInterface import ParserInterface

class NumpyDataTypeEncoder(json.JSONEncoder):
    def default(self, np_data):
        if isinstance(np_data, np.floating):
            return float(np_data)
        elif isinstance(np_data, np.ndarray):
            return np_data.tolist()
        elif isinstance(np_data, np.integer):
            return int(np_data)
        else:
            return super(NumpyDataTypeEncoder, self).default(np_data)

class RuntimeParser(ParserInterface):
    """
    This parser parses the CSV of per-layer runtime metrics generated by qnn-profile-viewer.
    The parser expects the CSV file path to be passed in via argv.
    """

    def __init__(self):
        """
        Instantiates a runtime parser, which maintains dictionaries of processes that are not
        broken down by layer (just_root) and processes that have both netrun and backend runtimes
        displayed as timing sources for root events (netrun_and_backend). For processes with both
        netrun and backend timing sources, we choose to consider the backend runtime.
        """
        self.root_and_subevent = {'EXECUTE'}
        self.netrun_and_backend = {'FINALIZE', 'DE-INIT', 'EXECUTE'}

    @staticmethod
    def remove_netrun(dataframe):
        """
        This function filters out from a dataframe rows for which 'backend' is not the listed timing source.
        :param dataframe: dataframe to filter
        :return: the filtered dataframe
        """
        return dataframe[dataframe['Timing Source'] == 'BACKEND']

    def add_value_to_dict(self,runtime_dict,event,value,unit):
        if event not in runtime_dict:
            runtime_dict[event]={}
            runtime_dict[event]["value"] = value;
            runtime_dict[event]["unit"] = unit;
            runtime_dict[event]["avg"] = value;
        else:
            if type(runtime_dict[event]["value"]) is not list:
                runtime_dict[event]["value"] = [runtime_dict[event]["value"]]
                runtime_dict[event]["value"].append(value)
                runtime_dict[event]["avg"]=np.mean(runtime_dict[event]["value"])

    def parse(self, file):
        """
        This method parses and summarizes the runtime outputs for each of the following processes:
        INIT, COMPOSE GRAPHS, FINALIZE, DE-INIT, EXECUTE.
        :param file: path to the CSV file to parse
        :return: a dictionary of length five; the keys are the process names; the values are sub-dictionaries;
        the sub-dictionaries' keys are 'root' and (if sub-events exist) layer names, and the values are the
        runtime values
        """
        # skips header rows and removes extra whitespace from column names
        df = pd.read_csv(file, skiprows=3)
        df.rename(columns=lambda x: x.strip(), inplace=True)
        process_to_df = {process: df for process, df in df.groupby('Message')}
        root_summary = {}
        sub_summary = {}
        for process, df in process_to_df.items():
            df.reset_index(inplace=True, drop=True)
            if process in self.netrun_and_backend:
                df = self.remove_netrun(df)
                process_to_df[process] = df
            if process in self.root_and_subevent:
                root_dict = {}
                subevent_dict = {}
                for ind in df.index:
                    # event is either root, or is the identifier (layer) of a sub-event,
                    # Use this name '__root__' to avoid conflicts with layer names
                    # (if it's a layer, we remove a leading underscore from the string)
                    time_unit = df['Unit of Measurement'][ind]
                    runtime = df['Time'][ind]
                    if df['Event Level'][ind] == "ROOT":
                        if pd.isna(df['Event Identifier'][ind]):
                            event = "root"
                        else:
                            event = df['Event Identifier'][ind]
                        self.add_value_to_dict(root_dict,event,runtime,time_unit)
                    elif df['Event Level'][ind] == "SUB-EVENT":
                        if df['Event Identifier'][ind][0]=="_" :
                            event = df['Event Identifier'][ind][1:]
                        else:
                            event = df['Event Identifier'][ind].split(":")[0]
                        self.add_value_to_dict(subevent_dict,event,runtime,time_unit)
                root_summary[process] = root_dict;
                sub_summary[process] = subevent_dict;
            else:
                runtime_dict = {}
                for ind in df.index:
                    time_unit = df['Unit of Measurement'][ind]
                    runtime = df['Time'][ind]
                    if pd.isna(df['Event Identifier'][ind]):
                        event = "root"
                    else:
                        event = df['Event Identifier'][ind]
                    self.add_value_to_dict(runtime_dict,event,runtime,time_unit)
                root_summary[process] = runtime_dict


        #since Execute reports runtime for each input, we also compute
        #the average per-layer runtime for execution
        #print(root_summary)
        return root_summary,sub_summary


    def diff(self,file1,file2):
        """
        This method parses two csv files and subtracts the result of first file with the second file.
        A positive result indicates that the first model is faster and corresponds to a smaller time value,
        and a negative value is the opposite.
        :param file1: path to the first CSV file to parse
        :param file2: path to the second CSV file to parse
        :return: a dictionary, the keys are '__root__' and layer names; For root ,the values are COMPOSE GRAPHS/
        INIT/DE-INIT/EXECUTE time differences for whole graph.
        For layers (if sub-events exist), the values are EXECUTE/FINALIZE(if exist) time differences for this layer.
        """
        root_summary1,sub_summary1=self.parse(file1)
        root_summary2,sub_summary2=self.parse(file2)
        diff_root_summary={}
        diff_sub_summary={}
        diff_summary={}

        for process, df in sub_summary1.items():
            assert (len(sub_summary1[process])==len(df)), "The two files have different structures"
            for event in df:
                if event not in sub_summary1:
                    diff_sub_summary[event] = {}
                    diff_sub_summary[event][process]={}
                    assert (sub_summary1[process][event]['unit'] == sub_summary2[process][event]['unit']), "The two files have different time unit"
                    diff_sub_summary[event][process]["value"]= (sub_summary2[process][event]['avg'] - sub_summary1[process][event]['avg'])/(sub_summary1[process][event]['avg']+1E-10);
                    diff_sub_summary[event][process]["unit" ]= sub_summary2[process][event]['unit']
                    #runtime_dict[event]

        for process, df in root_summary1.items():
            assert (len(root_summary2[process])==len(df)), "The two files have different structures"
            for event in df:
                if event not in diff_root_summary:
                    diff_root_summary[event] = {}
                if process not in diff_root_summary[event]:
                    diff_root_summary[event][process]={}
                assert (root_summary1[process][event]['unit'] == root_summary2[process][event]['unit']), "The two files have different time unit"
                diff_root_summary[event][process]["value"]= (root_summary2[process][event]['avg'] - root_summary1[process][event]['avg'])/(root_summary1[process][event]['avg']+1E-10);
                diff_root_summary[event][process]["unit" ]= root_summary1[process][event]['unit']
                    #runtime_dict[event]
        diff_summary=diff_sub_summary;
        diff_summary["__root__"]=diff_root_summary;
        result=json.dumps(diff_summary,cls=NumpyDataTypeEncoder)
        print(result)
        return diff_summary


if __name__ == '__main__':
    csv_file = sys.argv[1]
    csv2_file = sys.argv[2]
    runtimeParser= RuntimeParser()
    runtimeParser.diff(csv_file,csv2_file)
